﻿/*
 * ZPG semestralni prace
 * Petr Vondras, A06564
 * 2007
 */
using System;
using System.IO;
using System.Data;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using D3D=Microsoft.DirectX.Direct3D;

namespace Zpg
{
	public class ZpgD3D : Form
	{
		#region Globalni konstanty
    	/// <summary>Pocet bodu sceny v jednom smeru.</summary>
    	private const short VELIKOST = 128;
    	
    	/// <summary>Vzdalenost bodu sceny v metrech.</summary>
    	private const float ROZTEC = 2f;
    	
    	/// <summary>Rychlost pohybu v metrech za sekundu.</summary>
    	private const float RYCHLOST = 3;
    	
    	/// <summary>Vyska postavy v metrech.</summary>
    	private const float VYSKA = 1.85f;    	
    	
    	/// <summary>Delka dne (12h) v sekundach.</summary>
    	private const float DEN = 60f; 
    	
        /// <summary>Koeficient zpomaleni do kopce a zrychleni z kopce.
        /// hodnota 0 az 1 (0 = zadne zpomaleni).</summary>
        private const float ZPOMALENI = 0.5f;      	
        
        /// <summary>Citlivost mysi - vyssi cislo = mensi citlivost.</summary>
        private const float CITLIVOST = 500.0f;    			
		#endregion Globalni konstanty
		
		#region Globalni promenne
		/// <summary>Yaw - rotace podle vertikalni osy - rozhlizeni.</summary>
        private float yaw = 0;

        /// <summary>Pitch - rotace podle horizontalni osy - koukani nahoru a dolu.</summary>
        private float pitch = 0;
        private bool inveratace = true;
        
        /// <summary>Pozastavit zobrazeni slunce (slunce dal obyha).</summary>
        private bool pohybSlunce = true;
        
        /// <summary>Zapnuti - vypnuti baterky.</summary>
        private bool baterka = true;
        
        /// <summary>Vektor posunu pozice stredu (moji pozice).</summary>
        private Vector3 posun = new Vector3(-(VELIKOST-1)*ROZTEC/2, -(VELIKOST-1)*ROZTEC/2, 0);
        
        /// <summary>Vektor pohybu vpred.</summary>
        private short pohybVpred = 0; 
		
        /// <summary>Vektor pohybu bokem.</summary>
        private short pohybUkrok = 0;         
        
        /// <summary>Direct3D device.</summary>		
		private Device device = null;
		
		/// <summary>Direct3D vertx buffer.</summary>
		private VertexBuffer vertexBuffer = null;
		
		/// <summary>Direct3D index buffer.</summary>
		private IndexBuffer indexBuffer = null;		
		
		/// <summary>Direct3D mesh.</summary>
		private Mesh meshTeren = null;
		
		private Texture texturaTerenu = null;
		
		/// <summary>Vyskova mapa.</summary>
		private float[,] heightData;
		
		/// <summary>Pole vrcholu.</summary>
		private CustomVertex.PositionTextured[] verts;
		
		/// <summary>Pole trojuhelniku.</summary>
		private short[] indices;		
		
	    /// <summary>
        /// Pocet tiku od minuleho framu.
        /// </summary>
        private long tiku = 0;					//
        private int pocetTiku = 1;				//
        private int Frames = 0;					//
        private float snimkovaciFrekvence = 0;   // fps behem minule vteriny
        private D3D.Font text;					//
        
        // skybox
        private Mesh meshSkyboxu;
        private Material[] materialySkyboxu;
        private Texture[] texturySkyboxu;        
		#endregion Globalni promenne
		
		/// <summary>Konstruktor.</summary>
		public ZpgD3D()
		{
            // nastaveni parametru okna
            this.Size = new Size(800, 600);
            this.Text = "Petr Vondraš - ZPG Semestrální práce";
            
            // pohyb mysi - event handler
            this.MouseMove += new MouseEventHandler(PohybMysi);
            // stisk klavasy - event handler
            this.KeyDown += new KeyEventHandler(StiskKlavesy);			
       		// uvolneni klavasy - event handler
            this.KeyUp += new KeyEventHandler(UvolneniKlavesy);	            
		}
		
		/// <summary>Nacteni vyskovych dat.</summary>       
		private void NacistVysky() {
             heightData = new float[VELIKOST,VELIKOST];
             FileStream fs = new FileStream(Application.StartupPath + @"\mapa128x128.raw",FileMode.Open, FileAccess.Read);
             BinaryReader r = new BinaryReader(fs);
             
             for (int i = 0;i< VELIKOST;i++) {
             	for (int y=0; y< VELIKOST; y++) {
                    heightData[VELIKOST-1-y,VELIKOST-1-i] = (int)(r.ReadByte()/10);;
                }
             }
             r.Close();
        }   
		
		/// <summary>Nastaveni parametru grafiky.</summary>
		public bool InicializaceGrafiky()
		{
			try
			{
				#region nastaveni parametru
				PresentParameters presentParams = new PresentParameters();
				// neni fullscreen
				presentParams.Windowed=true; 
				// vyradit framy
				presentParams.SwapEffect = SwapEffect.Discard;
				// yapnout depth stencil
				presentParams.EnableAutoDepthStencil = true; 
				// nastavit stencil format
				presentParams.AutoDepthStencilFormat = DepthFormat.D16; 			
				// zjisteni a nastaveni schopnosti graficke karty
                Caps caps = Manager.GetDeviceCaps(0, Microsoft.DirectX.Direct3D.DeviceType.Hardware);
                CreateFlags flags;
                if (caps.DeviceCaps.SupportsHardwareTransformAndLight)
                    flags = CreateFlags.HardwareVertexProcessing;
                else
                    flags = CreateFlags.SoftwareVertexProcessing;
                // vytvorit D3D device
                device = new Microsoft.DirectX.Direct3D.Device(0,
                	Microsoft.DirectX.Direct3D.DeviceType.Hardware, this, flags, presentParams);
				#endregion nastaveni parametru
				
				#region nastaveni device
                // vypnout cull - uvidime predni i zadni cast trojuhelniku
				device.RenderState.CullMode = Cull.CounterClockwise;
				// zapnout ZBuffer
				device.RenderState.ZBufferEnable = true;
				// zkontrolovat zapnita svetla
				device.RenderState.Lighting = true;
				// priradit textury
				texturaTerenu = TextureLoader.FromFile(device, Application.StartupPath + @"\texturaTeren.jpg");
				device.SetTexture(0,texturaTerenu);
				// nastaveni vlastnosti textury
				device.TextureState[0].ColorOperation = TextureOperation.Modulate;
				device.TextureState[0].ColorArgument1 = TextureArgument.TextureColor;
				device.TextureState[0].ColorArgument2 = TextureArgument.Diffuse;
				device.TextureState[0].AlphaOperation = TextureOperation.Disable;
				// rozmazani textury
             	device.SamplerState[0].MinFilter = TextureFilter.Linear;
             	device.SamplerState[0].MagFilter = TextureFilter.Linear;
 				// zamaskovani linek prechodu mezi texturami
             	device.SamplerState[0].AddressU = TextureAddress.Mirror;
             	device.SamplerState[0].AddressV = TextureAddress.Mirror;
				// naplnit VB
				vertexBuffer = new VertexBuffer(typeof(CustomVertex.PositionTextured), VELIKOST*VELIKOST,
					device, Usage.WriteOnly, CustomVertex.PositionTextured.Format, Pool.Default);
				this.VytvoritVertexBuffer(vertexBuffer);
				// naplnit IB
				indexBuffer = new IndexBuffer(typeof(short), (VELIKOST-1)*(VELIKOST-1)*6,
					device, Usage.WriteOnly, Pool.Default);
				this.VytvoritIndexBuffer(indexBuffer);
				this.VytvoritMesh();
				this.VytvoritSkybox();
                #endregion nastaveni device

				return true;
			}
            catch (DirectXException) {
                MessageBox.Show("Selhala inicializace direct3D.", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }
		}
		
		/// <summary>Naplneni vertex bufferu daty.</summary>
		public void VytvoritVertexBuffer(object sender)
		{
			int indexTextury;
			VertexBuffer vb = (VertexBuffer)sender;
			// zamknuti
			verts = (CustomVertex.PositionTextured[])vb.Lock(0,0);
			// naplneni
            for (int x=0;x< VELIKOST;x++)
            {
            	for (int y=0; y< VELIKOST;y++)
            	{
            		verts[x+y*VELIKOST].X=ROZTEC*x;
            		verts[x+y*VELIKOST].Y=ROZTEC*y;
            		verts[x+y*VELIKOST].Z=heightData[x,y];
            		if (((double)x/2) == (Math.Truncate((double)x/2))) verts[x+y*VELIKOST].Tu = 0;
            		else verts[x+y*VELIKOST].Tu = 1;
            		indexTextury = (int)(Math.Truncate(heightData[x,y]));
            		if (((double)y/2) == (Math.Truncate((double)y/2)))
            		{
            			if ((y == 0)||(verts[x+(y-1)*VELIKOST].Tv != indexTextury/26f))
            			{
							verts[x+y*VELIKOST].Tv = indexTextury/26f;
            				verts[x+(y+1)*VELIKOST].Tv = (indexTextury+1)/26f;            				
            			}
            			else
            			{
							verts[x+y*VELIKOST].Tv = (indexTextury-1)/26f;
            				verts[x+(y+1)*VELIKOST].Tv = (indexTextury)/26f;
            			}
            		}	
                }
            } 
			// odemknuti
			vb.Unlock();
		}
		
		/// <summary>Naplneni index bufferu daty.</summary>
		public void VytvoritIndexBuffer(object sender)
		{
			IndexBuffer ib = (IndexBuffer)sender;
			// zamknuti
			indices = (short[])ib.Lock(0,0);
			// naplneni
            for (int x=0;x< VELIKOST-1;x++)
            {
            	for (int y=0; y< VELIKOST-1;y++)
            	{
                	indices[(x+y*(VELIKOST-1))*6] = (short) ((x+1)+(y+1)*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+2] = (short) ((x+1)+y*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+1] = (short) (x+y*VELIKOST);
                	
                		
                	indices[(x+y*(VELIKOST-1))*6+3] = (short) ((x+1)+(y+1)*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+5] = (short) (x+y*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+4] = (short) (x+(y+1)*VELIKOST);
                	
                }
            }
			// odemknuti
			ib.Unlock();
		}		
				
		/// <summary>Naplneni meshe stream daty.</summary>
        private void VytvoritMesh()
        {
            meshTeren = new Mesh((VELIKOST - 1) * (VELIKOST - 1) * 2, VELIKOST * VELIKOST, MeshFlags.Managed, CustomVertex.PositionTextured.Format, device);
 
            meshTeren.SetVertexBufferData(verts, LockFlags.None);
            meshTeren.SetIndexBufferData(indices, LockFlags.None);
 
            int[] adjac = new int[meshTeren.NumberFaces * 3];
            meshTeren.GenerateAdjacency(5.5f, adjac);
            meshTeren.OptimizeInPlace(MeshFlags.OptimizeVertexCache, adjac);
            
            meshTeren = meshTeren.Clone(meshTeren.Options.Value, CustomVertex.PositionNormalTextured.Format, device);
            meshTeren.ComputeNormals();
        }
        
		/// <summary>Naplneni meshe stream daty.</summary>
        private void VytvoritSkybox()
        {
        	ExtendedMaterial[] poleMaterialu;
            meshSkyboxu = Mesh.FromFile("skybox.x", MeshFlags.Managed, device, out poleMaterialu);
 
            if ((poleMaterialu != null) && (poleMaterialu.Length > 0))
            {
                materialySkyboxu = new Material[poleMaterialu.Length];
                texturySkyboxu = new Texture[poleMaterialu.Length];
 
                for (int i = 0; i < poleMaterialu.Length; i++)
                {
                    materialySkyboxu[i] = poleMaterialu[i].Material3D;
                    materialySkyboxu[i].Ambient = materialySkyboxu[i].Diffuse;
                    
                    if ((poleMaterialu[i].TextureFilename != null) && (poleMaterialu[i].TextureFilename != string.Empty))
                    {
                        texturySkyboxu[i] = TextureLoader.FromFile(device, poleMaterialu[i].TextureFilename);
                    }
                } 
            }
            meshSkyboxu = meshSkyboxu.Clone(meshSkyboxu.Options.Value, CustomVertex.PositionNormalTextured.Format, device);
            meshSkyboxu.ComputeNormals();
        }        
		         	
		/// <summary>Nastaveni matic projekce, sveta a kamery.</summary>
		private void NastaveniMatic()
		{
			// posun sveta dle nasi pozice
		    device.Transform.World = Matrix.Translation(posun);  
			
			// nastaveni kamery a natoceni pohledu
			device.Transform.View = Matrix.LookAtLH(new Vector3(0,0,0), new Vector3(0,1,0), new Vector3(0,0,1));
            device.Transform.View *= Matrix.RotationY(yaw);
            device.Transform.View *= Matrix.RotationX(pitch);    
            
			// perspektivni transformace
			// + nastaveni hlobky viditelnosti
			float aspect = (float)this.ClientSize.Width / (float)this.ClientSize.Height;
            device.Transform.Projection = Matrix.PerspectiveFovLH((float)Math.PI / 4, aspect, 0.1f, 200.0f);	         
		}
		
		/// <summary>Nastaveni vlastnosti svetel.</summary>
		private void NastaveniSvetel()
		{
			System.Drawing.Color col = System.Drawing.Color.White;
			// nastavit material terenu.
			// v jednom case muze byt pouzit pouze jeden material
			D3D.Material mtrl = new D3D.Material();
			mtrl.Diffuse = col;
			mtrl.Ambient = col;
			device.Material = mtrl;
			
			// nastavit slunce.
			// v nastaveni grafiky je treba enablovat D3DRS_LIGHTING
			if (pohybSlunce)
			{
				device.Lights[0].Type = LightType.Directional;
				device.Lights[0].Diffuse = System.Drawing.Color.White;
				device.Lights[0].Direction =
					new Vector3((float)Math.Cos((Math.PI*Environment.TickCount) / (1000*DEN)),
				            0.0f, -(float)Math.Sin((Math.PI*Environment.TickCount) / (1000*DEN)));
				// dat svetlo na vedomi grafice
				device.Lights[0].Commit();
				// zapnout svetlo				
			}
			device.Lights[0].Enabled = true;
			
			// nasavit baterku.
			if (baterka)
			{
				device.Lights[1].Type = LightType.Spot;
				device.Lights[1].Diffuse = System.Drawing.Color.White;
				device.Lights[1].Position = new Vector3(0, 0, 0f);
				device.Lights[1].Direction = new Vector3((float)Math.Cos(pitch)*(float)Math.Sin(yaw), 
			                                         	 (float)Math.Cos(pitch)*(float)Math.Cos(yaw), 
			                                         	 (float)Math.Sin(pitch));
				device.Lights[1].InnerConeAngle = 0.5f;
				device.Lights[1].OuterConeAngle = 1f;
				device.Lights[1].Falloff = 2.0f;
				device.Lights[1].Attenuation0 = 1f;
				device.Lights[1].Range = 100f;
				device.Lights[1].Commit();
				device.Lights[1].Enabled = true;				
			}
			else device.Lights[1].Enabled = false;
		}
		
		/// <summary>Vypocet posunu po terenu.</summary>
		private void Pohyb()
		{	
            // vypocet uplynuleho casu
            long noveTiku = Environment.TickCount;
            float uplynulyCas = noveTiku - tiku;
            tiku = noveTiku;
            
            if ((pohybVpred != 0)||(pohybUkrok != 0))
            {
            	Vector3 V1,V2,V3;
				/// <summary>Vektor smeru pohybu.</summary>
         		Vector3 pohyb = new Vector3((float) Math.Sin(yaw)*pohybVpred - (float) Math.Cos(yaw)*pohybUkrok,
                                            (float) Math.Cos(yaw)*pohybVpred + (float) Math.Sin(yaw)*pohybUkrok,
                                            0);
            	#region Posun ve vodorovnem smeru
            	pohyb.Normalize();
            	pohyb.Multiply(RYCHLOST*uplynulyCas/1000);
            
            	// kontrola hranic
            	if ((posun.X - pohyb.X) > -5) pohyb.X = posun.X +5;
            	if ((posun.X - pohyb.X) < (5-(VELIKOST-1)*ROZTEC)) pohyb.X = posun.X - (5-(VELIKOST-1)*ROZTEC);
            	if ((posun.Y - pohyb.Y) > -5) pohyb.Y = posun.Y +5;
            	if ((posun.Y - pohyb.Y) < (5-(VELIKOST-1)*ROZTEC)) pohyb.Y = posun.Y - (5-(VELIKOST-1)*ROZTEC);
            	#endregion Posun ve vodorovnem smeru

            	#region Vypocet vysky
            	// vrcholy trojuhelnika nad kterym se nachazime
            	Vector3 newPosun = new Vector3(posun.X - pohyb.X, posun.Y - pohyb.Y, 0);
            	int pomocna = (int) (Math.Truncate(-newPosun.X/ROZTEC)+1+(Math.Truncate(-newPosun.Y/ROZTEC)+1)*VELIKOST);
            	V1 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            	if ((-newPosun.X/2 - Math.Truncate(-newPosun.X/ROZTEC)) < (-newPosun.Y/ROZTEC - Math.Truncate(-newPosun.Y/ROZTEC))) {
            		pomocna = (int) (Math.Truncate(-newPosun.X/ROZTEC)+Math.Truncate(-newPosun.Y/ROZTEC)*VELIKOST);
            		V2 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            		pomocna = (int) (Math.Truncate(-newPosun.X/ROZTEC)+(Math.Truncate(-newPosun.Y/ROZTEC)+1)*VELIKOST); 
            		V3 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            	}
            	else {
            		pomocna = (int) (Math.Truncate(-newPosun.X/ROZTEC)+1+Math.Truncate(-newPosun.Y/ROZTEC)*VELIKOST); 
            		V2 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            		pomocna = (int) (Math.Truncate(-newPosun.X/ROZTEC)+Math.Truncate(-newPosun.Y/ROZTEC)*VELIKOST); 
            		V3 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            	}
            
            	// vypocet z - souradnice pruseciku plochy trojuhelnika
            	// s rovnobezkou s osou z prochazejici nasi pozici
            	// a zvednuti o vyku postavy nad teren
            	newPosun.Z = - VYSKA - V1.Z
            		- (((V2.Y-V1.Y)*(V3.Z-V1.Z)-(V2.Z-V1.Z)*(V3.Y-V1.Y))*(V1.X+newPosun.X)
            		+ ((V3.X-V1.X)*(V2.Z-V1.Z)-(V3.Z-V1.Z)*(V2.X-V1.X))*(V1.Y+newPosun.Y))
            		/((V2.X-V1.X)*(V3.Y-V1.Y)-(V2.Y-V1.Y)*(V3.X-V1.X));
            	pohyb.Z = posun.Z - newPosun.Z ;
            	#endregion Vypocet vysky
            
				pohyb.Normalize();
            	// zpomaleni do kopce zrychleni z kopce
            	pohyb.Multiply((float) Math.Pow(1-pohyb.Z, ZPOMALENI));
            	// rychlost
            	pohyb.Multiply(RYCHLOST*uplynulyCas/1000);
            	posun.Subtract(pohyb);
            }
		}
		
		/// <summary>Vypocet pocatecni vysky.</summary>
		private void PocatecniVyska()
		{	
            	Vector3 V1,V2,V3;
            	// vrcholy trojuhelnika nad kterym se nachazime
            	int pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+1+(Math.Truncate(-posun.Y/ROZTEC)+1)*VELIKOST);
            	V1 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            	if ((-posun.X/2 - Math.Truncate(-posun.X/ROZTEC)) < (-posun.Y/ROZTEC - Math.Truncate(-posun.Y/ROZTEC))) {
            		pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST);
            		V2 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            		pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+(Math.Truncate(-posun.Y/ROZTEC)+1)*VELIKOST); 
            		V3 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            	}
            	else {
            		pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+1+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST); 
            		V2 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            		pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST); 
            		V3 = new Vector3(verts[pomocna].X, verts[pomocna].Y, verts[pomocna].Z);
            	}
            
            	// vypocet z - souradnice pruseciku plochy trojuhelnika
            	// s rovnobezkou s osou z prochazejici nasi pozici
            	// a zvednuti o vyku postavy nad teren
            	posun.Z = - VYSKA - V1.Z
            		- (((V2.Y-V1.Y)*(V3.Z-V1.Z)-(V2.Z-V1.Z)*(V3.Y-V1.Y))*(V1.X+posun.X)
            		+ ((V3.X-V1.X)*(V2.Z-V1.Z)-(V3.Z-V1.Z)*(V2.X-V1.X))*(V1.Y+posun.Y))
            		/((V2.X-V1.X)*(V3.Y-V1.Y)-(V2.Y-V1.Y)*(V3.X-V1.X));
		}		
		
		/// <summary>Vykresleni sceny.</summary>
		private void Render()
		{
			if (device == null) return;
			// vycistit backbuffer na modro a nastavit depth buffer
			device.Clear(ClearFlags.Target | ClearFlags.ZBuffer, Color.DarkSlateBlue, 1f, 0);
			// zacatek sestaveni sceny
			device.BeginScene();
			// skybox
			device.Lights[0].Enabled = false;
			device.Lights[1].Enabled = false;
			device.Transform.World = Matrix.Translation(0,0,0);
			short barva = (short) Math.Truncate(255 * Math.Sin((Math.PI*Environment.TickCount)/ (1000*DEN)));
			if (barva < 0) barva = 0;
			device.RenderState.Ambient = Color.FromArgb(barva,barva,barva);
            for (int i = 0; i < materialySkyboxu.Length; i++)
            {
                device.SetTexture(0, texturySkyboxu[i]);
                meshSkyboxu.DrawSubset(i);
            }  
			// nastavit svetla a materialy
			NastaveniSvetel();
			// posun po terenu
			Pohyb();			
			// nastavit matice sveta, pohledu a projekce
			NastaveniMatic();
			// trojuhelnikova mapa
			//device.SetStreamSource(0, vertexBuffer, 0);
			//device.VertexFormat = CustomVertex.PositionTextured.Format;
			//device.Indices = indexBuffer;			
			//device.DrawIndexedPrimitives(PrimitiveType.TriangleList, 0, 0, VELIKOST*VELIKOST, 0, (VELIKOST-1)*(VELIKOST-1)*2);
			//device.RenderState.FillMode = FillMode.WireFrame;
			
			// vykreslit mesh terenu
			device.SetTexture(0, texturaTerenu);
			int numSubSets = meshTeren.GetAttributeTable().Length;
            for (int i = 0; i < numSubSets; ++i)
            {
                meshTeren.DrawSubset(i);
            }

		    // zobrazit framerate - nakonec, aby byl navrch
		    FrameRate();
            text.DrawText(null,string.Format("Framerate : {0:0.00} fps", snimkovaciFrekvence), 
            	new Rectangle(10, 10,700,200),Microsoft.DirectX.Direct3D.DrawTextFormat.Left, Color.Red);			
			// konec sestaveni
			device.EndScene();
			// zobrazeni vysledku
			device.Present();		
		}
		
		/// <summary>Vypocet framerate - ulozeno do promenne snimkovaci frekvence.</summary>
		private void FrameRate()
		{
		    Frames++;
            if (Math.Abs(Environment.TickCount - pocetTiku) > 1000) {
            	snimkovaciFrekvence = (float)Frames * 1000 / Math.Abs(Environment.TickCount - pocetTiku);
                pocetTiku = Environment.TickCount;
                Frames = 0;
            }
		}
		
        /// <summary>Stisk klavasy - event handler.</summary>
        private void StiskKlavesy(object sender, KeyEventArgs e) {
            switch (e.KeyCode) {
                case Keys.Escape:
                    // ukonci aplikaci
                    this.Close();
                    break;
                case Keys.W:
                    // pohyb vpred
                    pohybVpred = 1;
                    break;
                case Keys.S:
                    // pohyb vzad
                    pohybVpred = -1;     
                    break;
                case Keys.D:
                    // pohyb vpravo
                    pohybUkrok = 1;
                    break; 
                case Keys.A:
                    // pohyb vlevo
                    pohybUkrok = -1;
                    break;
                case Keys.U:
                    if (inveratace)	inveratace = false;
                    else inveratace = true;
                    break;
                case Keys.B:
                    // baterka
                    if (baterka) baterka = false;
                    else baterka = true;
                    break;                                          
                case Keys.Space:
                    // nastavit vychozi hodnoty
                    posun.X = -VELIKOST/2*ROZTEC;
                    posun.Y = -VELIKOST/2*ROZTEC;
                    PocatecniVyska();
                    yaw = 0;
                    pitch = 0;
                    break;
                default:
                    break;
            }
        }   
        
        /// <summary>Uvolneni klavasy - event handler.</summary>
        private void UvolneniKlavesy(object sender, KeyEventArgs e) {
            switch (e.KeyCode) {
                case Keys.W:
                    // pohyb vpred
                    pohybVpred = 0;
                    break;
                case Keys.S:
                    // pohyb vzad
                    pohybVpred = 0;            
                    break;
                case Keys.D:
                    // pohyb vpravo
                    pohybUkrok = 0;
                    break; 
                case Keys.A:
                    // pohyb vlevo
                    pohybUkrok = 0;
                    break;
                default:
                    break;
            }
        }           
        
        /// <summary>Pohyb mysi - event handler.</summary>
        private void PohybMysi(object sender, MouseEventArgs e) {
            // vypocet stredu okna
            Size s = this.ClientSize;
            int cx = s.Width / 2;
            int cy = s.Height / 2;

            // kontrola zda se pohnulo s mysi
            if (e.X == cx && e.Y == cy) return;

            // aktualizace uhlu
            yaw += (float)(cx - e.X) / CITLIVOST;
            if (inveratace)
            	pitch += (float)(cy - e.Y) / CITLIVOST;
            else pitch -= (float)(cy - e.Y) / CITLIVOST;

            // kontrola rozsahu uhlu
            if (yaw > 2 * Math.PI) yaw -= 2 * (float)Math.PI;
            if (pitch > Math.PI / 2) pitch = (float)Math.PI / 2;
            else if (pitch < -Math.PI / 2) pitch = -(float)Math.PI / 2;

            // vypocitat pozici stredu rel. k desktopu z pozice stredu rel. k soucasnemu oknu
            Point center = PointToScreen(new Point(cx, cy));
            // vycentrovat kurzor
            Cursor.Position = center;
        }              
     
        /// <summary>Vlastni spusteni.</summary>
        private void Run() {
            // schovat kurzor mysi
            Cursor.Hide();
            // zjistit velikost okna a spocitat stred
            Size s = this.ClientSize;
            int cx = s.Width / 2;
            int cy = s.Height / 2;
            // vypocitat pozici stredu rel. k desktopu z pozice stredu rel. k soucasnemu oknu
            Point center = PointToScreen(new Point(cx, cy));
            // vycentrovat kurzor
            Cursor.Position = center;
            // nastavit cas
            tiku = Environment.TickCount;
            // priprava zobrazovaneho textu - musi byt po inicializaci device
            System.Drawing.Font systemfont = new System.Drawing.Font("Arial", 12f, FontStyle.Bold);
            text = new D3D.Font(device,systemfont);
            // vypocet pocatecni vysky
            PocatecniVyska();            

            // behova smycka
            while (this.Created) {
                Render();
                Application.DoEvents(); // melo by byt posledni ve smycce
            }

            // zobrazit kurzor mysi
            Cursor.Show();
            // zrusit zinicializovane
            if (vertexBuffer != null) vertexBuffer.Dispose();
            if (indexBuffer != null) indexBuffer.Dispose();
            if (device != null) device.Dispose();
        }		
        
        /// <summary>Start aplikace.</summary>
		static void Main() 
		{
			// vytvoreni okna aplikace
			ZpgD3D form = new ZpgD3D();
            // zobrazeni okna
            form.Show();
            // nacteni vyskove mapy
            form.NacistVysky();
            // inicializace Direct 3D
            form.Text += " - Načítáni dat.";
            if (form.InicializaceGrafiky()){
            	// vlastni spusteni
            	form.Text = "Petr Vondraš - ZPG Semestrální práce";
                form.Run();
            }   
		}
	}
}
