using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using System.Windows.Forms;
using System.Drawing;

namespace zpg
{
    class Terrain:GraphicObject
    {
        /**
         * Create the terrain
         */
        public Terrain(Device graphicsDevice )
            :base(graphicsDevice)
        {   
            rawData = NormalizeHeightMap(
                Squarify(
                Load(new FileInfo(Properties.Settings.Default.terrainFile))));
            graphicsStreamIndex = getFreeStreamIndex();
            CreateVertexBuffer();
        }

        /**
         * Draw the terrain
         */
        protected override void Render(GraphicObjectView view)
        {
            graphicsDevice.SetStreamSource( graphicsStreamIndex, vertexBuffer, 0);
            graphicsDevice.VertexFormat = CustomVertex.PositionNormal.Format;
            graphicsDevice.DrawPrimitives(PrimitiveType.TriangleStrip, 0, triangleCount);
        }

        /**
         * Return true if given object is in projection of terrain into XZ plane.
         * Arguments are TERRAIN COORDINATES
         */
        public bool IsInTerrain(float x, float z)
        {
            x = x * rawData.Length;
            z = z * rawData[0].Length;

            int r = (int)Math.Floor(x);//nearest lowest index
            int c = (int)Math.Floor(z);

            if (r >= rawData.Length-1-borderWidth || r <= 0+borderWidth)
                return false;
            if (c >= rawData[0].Length - 1 - borderWidth || c <= 0+borderWidth)
                return false;
            return true;
        }

        /**
         * Compute elevation.
         * takes and returns values in TERRAIN COORDINATES, so if terrain is 
         * somehow scaled or moved, there must the some transformations applied to 
         * 
         */
        public float GetElevation(float Qx, float Qz)
        {
            int r = (int)Math.Floor(Qx  * rawData.Length);//nearest lowest index
            int c = (int)Math.Floor(Qz  * rawData[0].Length);
            if (r < 0 || c < 0 || r >= rawData.Length || c >= rawData[0].Length)
                return -1;

            Vector3 ground = new Vector3(rawData[r][c].X, rawData[r][c].Y, rawData[r][c].Z);
            Vector3 master = new Vector3(rawData[r+1][c+1].X, rawData[r+1][c+1].Y, rawData[r+1][c+1].Z);
            Vector3 flink;

            float sin45 = (float)Math.Sqrt(2) / 2;

            Vector3 Qbase = new Vector3(Qx, ground.Y, Qz);
            Vector3 QbaseRel = Vector3.Subtract(Qbase, ground);

            float sinQ = (Qx - ground.X) / QbaseRel.Length();

            if (sinQ < sin45)
            {
                flink = new Vector3(rawData[r][c+1].X, rawData[r][c+1].Y, rawData[r][c+1].Z);
            }
            else
            {
                flink = new Vector3(rawData[r+1][c].X, rawData[r+1][c].Y, rawData[r+1][c].Z);
            }


            Vector3 fvr = Vector3.Subtract(flink, ground);
            Vector3 mvr = Vector3.Subtract(master, ground);
            Vector3 normal = Vector3.Cross(fvr, mvr);

            float d =
                -Vector3.Dot(normal, ground);

            float Qy =
                (-d  - normal.X * Qx - normal.Z * Qz) / normal.Y;

            return Qy;
        }


        public override GraphicObjectView CreateView()
        {
            return new TerrainView( this );
        }

        private void  CreateVertexBuffer()
        {
            List<CustomVertex.PositionNormal> vertexes = new List<CustomVertex.PositionNormal>();
            for (int r = 0; r < rawData.Length - 1; r++)
            {
                for (int cc = 0; cc < rawData[r].Length - 2; cc++)
                {
                    for (int f = 0; f < 2; f++)
                    {
                        int c = (r % 2 == 0) ? cc : rawData[r].Length - 3 - cc;
                        Vector3 ground = rawData[r+f][c+f];
                        Vector3 normal = ComputeVertexNormalSomehow(r, c);
                        vertexes.Add(new CustomVertex.PositionNormal(ground, normal));
                    }
                }
            }

            //compute count of used triangles
            triangleCount = vertexes.Count - 3;
            
            vertexBuffer
                = new VertexBuffer(typeof(CustomVertex.PositionNormal),
            vertexes.Count,
            graphicsDevice,
            Microsoft.DirectX.Direct3D.Usage.WriteOnly,
            CustomVertex.PositionNormal.Format,
            Pool.Managed);

            CustomVertex.PositionNormal[] vb =
                 (CustomVertex.PositionNormal[])vertexBuffer.Lock(0, 0);
            {
                int vbi = 0;
                foreach (CustomVertex.PositionNormal v in vertexes)
                {
                    vb[vbi++] = v;
                }
            }
            vertexBuffer.Unlock();
        }

        private Vector3 ComputeVertexNormalSomehow(int r, int c)
        {
            Vector3 ground = rawData[r][c];
            int[,] order = new int[8,2] { 
                 {1,1}
                ,{0,1}
                ,{-1,1}
                ,{-1,0}
                ,{-1,-1}
                ,{0,-1}
                ,{1,-1}
                ,{1,0}
            };
            List<Vector3> surroundung = new List<Vector3>();

            for (int i = (order.Length / 2) -1; i>=0 ; i--)
            {
                int ro = order[i, 0];
                int co = order[i, 1];
                if ((r + ro > 0 && r + ro < rawData.Length)
                    && (c + co > 0 && c + co < rawData[r + ro].Length)
                    )
                {
                    surroundung.Add(
                        Vector3.Subtract(
                            rawData[r + ro][c + co]
                            , ground
                        )
                    );
                }
            }
            Vector3 normal = new Vector3(0, 0, 0);
            if (surroundung.Count != 0)
            {
                Vector3 lp = surroundung[0];
                surroundung.RemoveAt(0);
                foreach (Vector3 v in surroundung)
                {
                    Vector3 cross = Vector3.Cross(lp, v);
                    lp = v;
                    normal.Add(cross);
                }
            }
            normal.Normalize();
            return normal;
        }


        private Vector3 toVector(CustomVertex.PositionNormal src)
        {
            return new Vector3(src.X, src.Y, src.Z);
        }

        /**
        * load image data into one dimensional buffer
        */ 
        private byte[] Load(FileInfo datasource)
        {
            FileStream input = null;
            try
            {
                input = datasource.OpenRead();
                BinaryReader dataReader = new BinaryReader(input);
                byte[] buffer = new byte[ datasource.Length ];
                for (int i = 0; i < buffer.Length; i++)
                {
                    buffer[i] = dataReader.ReadByte();
                }
                return buffer;
            }
            finally
            {
                if (input != null)
                {
                    input.Close();
                }
            }
        }

        /**
         * best effort to create a square buffer
         */ 
        private byte[][] Squarify(byte[] buffer)
        {
            int rows = (int) Math.Sqrt(buffer.Length);
            int cols = (int) buffer.Length / rows;

            byte[][] result = new byte[rows][];
            {
                for (int r = 0; r < rows; r++)
                {
                    result[r] = new byte[cols];
                    for (int c = 0; c < cols; c++)
                    {
                        result[r][c] = buffer[ r * cols + c ];
                    }
                }
            }
            return result;   
        }

        /**
         * Normalize heightmap to <0,1> interval, compute vertices using real sizes
         */
        private Vector3[][] NormalizeHeightMap(byte[][] heightmap)
        {
            Vector3[][] terain//precompute the vertices
                = new Vector3[heightmap.Length][];
            for (int r = 0; r < heightmap.Length; r++)
            {
                terain[r] = new Vector3[heightmap[r].Length];
                for (int c = 0; c < heightmap[r].Length; c++)
                {
                    terain[r][c] = new Vector3();
                    terain[r][c].X = (r / (float)heightmap.Length);
                    terain[r][c].Y = (1 - heightmap[r][c] / (float)0xFF);
                    terain[r][c].Z = (c / (float)heightmap[r].Length);

                }
            }
            return terain;
        }

        public Size Resolution
        {
            get { return new Size(rawData.Length, rawData[0].Length); }
        }

        /**
         * VertexBuffer holding image data
         */
        private VertexBuffer vertexBuffer;
        /**
         * How close can avatar get to the border of the terrain
         */ 
        private int borderWidth = 2;

        /**
         * Number of triangles to draw
         */
        private int triangleCount;
        /**
         * Heightmap of the terrain
         */
        private Vector3[][] rawData;

        private int graphicsStreamIndex;
    }
}
