using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using System.Drawing;

namespace zpg
{
    
    class HeadsUpDisplay
    {
        private System.Drawing.Font font = new System.Drawing.Font(FontFamily.GenericSansSerif,16);

        public HeadsUpDisplay(Device graphicsDevice, Rectangle screen)
        {
            this.graphicsDevice = graphicsDevice;
            this.screen = screen;
        }

        public TextBox createTextBox(Rectangle targetRectangle, Color textColor, int capacity)
        {
            TextBox newBox = new TextBox(graphicsDevice, targetRectangle, textColor, font, capacity);
            textData.Add(newBox);
            return newBox;
        }

        public Console createConsole(Rectangle targetRectangle
            , Color textColor
            , Color backgroundColor
            , int capacity
            , CommandHandler handler)
        {
            Console newBox = new Console(graphicsDevice, targetRectangle, textColor, backgroundColor
                , font, capacity, handler, screen);
            textData.Add(newBox);
            return newBox;
        }


        public void Render()
        {

            foreach(TextBox textBox in textData)
            {
                textBox.Render();
            }
        }

        private List<TextBox> textData = new List<TextBox>();
        private Device graphicsDevice;
        private Rectangle screen;
    }

    class TextBox 
    {

        internal TextBox(Device graphicsDevice, Rectangle targetRectangle, Color textColor
            , System.Drawing.Font font, int capacity)
        {
            this.graphicsDevice = graphicsDevice;
            this.targetRectangle = targetRectangle;
            this.font = new Microsoft.DirectX.Direct3D.Font(graphicsDevice, font);

            this.textColor = textColor;
            this.capacity = capacity;
        }

        public void Clear()
        {
            textData.Clear();
        }

        public void AddText(String text)
        {
            textData.Add( text );
            if (textData.Count > capacity)
                textData.RemoveAt(0);
        }

        internal virtual void Render()
        {
            if (!Visible)
                return;
            Rectangle myrect = targetRectangle;
            foreach (String text in textData)
            {
                font.DrawText(null, text, myrect, drawTextFormat, textColor);
                myrect.Y -= font.Description.Height;
                myrect.Height -= font.Description.Height;
            }
        }

        private bool isVisible = true;

        public bool Visible
        {
            get { return isVisible; }
            set { isVisible = value; }
        }

        protected Rectangle targetRectangle;
        //private List<string> textData = new List<string>();
        protected Microsoft.DirectX.Direct3D.Font font;
        protected Color textColor = Color.White;
        protected List<string> textData = new List<string>();
        protected DrawTextFormat drawTextFormat = DrawTextFormat.NoClip | DrawTextFormat.ExpandTabs | DrawTextFormat.WordBreak;
        protected int capacity;
        protected Device graphicsDevice;
    }


    class Console : TextBox 
    {
        private CommandHandler commandHandler;
        private Color backgroundColor;
        private VertexBuffer background;
        private VertexBuffer frame;
        private Rectangle screen;

        private StringBuilder commandBuffer = new StringBuilder();

        internal Console(Device graphicsDevice
            , Rectangle targetRectangle
            , Color foregroundColor
            , Color backgroundColor
            , System.Drawing.Font font
            , int capacity
            , CommandHandler commandHandler
            , Rectangle screen)
            :base( graphicsDevice, targetRectangle, foregroundColor, font, capacity)
        {
            this.commandHandler = commandHandler;
            this.backgroundColor = backgroundColor;
            this.screen = screen;

            background = new VertexBuffer(
                typeof(CustomVertex.PositionColored),
                5,
                graphicsDevice,
                Usage.WriteOnly,
                CustomVertex.PositionColored.Format,
                Pool.Managed);

            CustomVertex.PositionColored[] vertices = (CustomVertex.PositionColored[])
            background.Lock(0, 0);
            vertices[0] = new CustomVertex.PositionColored(
                targetRectangle.Left + targetRectangle.Width, targetRectangle.Bottom + 5, 0f, backgroundColor.ToArgb());
            vertices[1] = new CustomVertex.PositionColored(
                targetRectangle.Left, targetRectangle.Bottom + 5, 0f, backgroundColor.ToArgb());
            vertices[2] = new CustomVertex.PositionColored(
                targetRectangle.Left + targetRectangle.Width, targetRectangle.Bottom - targetRectangle.Height, 0f, backgroundColor.ToArgb());
            vertices[3] = new CustomVertex.PositionColored(
                targetRectangle.Left, targetRectangle.Bottom - targetRectangle.Height, 0f, backgroundColor.ToArgb());
            background.Unlock();

            frame = new VertexBuffer(
                typeof(CustomVertex.PositionColored),
                5,
                graphicsDevice,
                Usage.WriteOnly,
                CustomVertex.PositionColored.Format,
                Pool.Managed);
            CustomVertex.PositionColored[] frameVerts = (CustomVertex.PositionColored[])
            frame.Lock(0, 0);

            frameVerts[0] = vertices[0];
            frameVerts[1] = vertices[1];
            frameVerts[2] = vertices[3];
            frameVerts[3] = vertices[2];
            frameVerts[4] = vertices[0];

            frame.Unlock();
            //clearscreen
            for (int i = 0; i < capacity; i++)
            {
                AddText("");
            }
        }

        internal override void Render()
        {
            if (!Visible)
                return;
            graphicsDevice.Transform.World = Matrix.Identity;
            graphicsDevice.Transform.View = Matrix.Identity;
            graphicsDevice.Transform.Projection
                = Matrix.OrthoOffCenterRH(screen.Left ,screen.Right ,screen.Bottom ,screen.Top, 0, 1);
             

            graphicsDevice.SetStreamSource(0, background, 0);
            graphicsDevice.VertexFormat = CustomVertex.PositionColored.Format;
            graphicsDevice.RenderState.FillMode = FillMode.Solid;
            graphicsDevice.RenderState.AlphaBlendEnable = true;
            graphicsDevice.RenderState.SourceBlend = Blend.One;
            graphicsDevice.RenderState.DestinationBlend = Blend.One;
            graphicsDevice.RenderState.BlendOperation = BlendOperation.Min;
            graphicsDevice.DrawPrimitives(PrimitiveType.TriangleStrip, 0, 2);
            graphicsDevice.RenderState.AlphaBlendEnable = false;

            graphicsDevice.SetStreamSource(0, frame, 0);
            graphicsDevice.VertexFormat = CustomVertex.PositionColored.Format;
            graphicsDevice.RenderState.FillMode = FillMode.WireFrame;
            graphicsDevice.DrawPrimitives(PrimitiveType.LineStrip, 0, 4);

            if( textData.Count > 0 )
                textData.RemoveAt(textData.Count - 1);

            textData.Add(commandBuffer.ToString());
            base.Render();
        }

        public void ProcessKey(Keys keyData)
        {
            switch (keyData)
            {
                case Keys.Enter:
                    if (commandBuffer.Length.ToString().Trim().Length > 0)
                    {
                        String response = commandHandler.ProcessCommand(textData[textData.Count - 1]);
                        AddText(response);
                        commandBuffer.Remove(0, commandBuffer.Length);
                        AddText(commandBuffer.ToString());
                    }
                break;
                case Keys.Back:
                    if( commandBuffer.Length > 0)
                        commandBuffer.Remove(commandBuffer.Length - 1, 1);
                break;
                default:
                    if(commandBuffer.Length < commandHandler.getMaxCommandLength())
                        commandBuffer.Append((char)keyData);
                break;
            }
        }

    }


    interface CommandHandler
    {
        /**
         * Handle command passed from console.
         * Return text to be displayed to user
         */
        String ProcessCommand(String command);

        /**
         * Return maximum length of command
         */
        int getMaxCommandLength();
    }

    class DummyCommandHandler : CommandHandler
    {

        public int getMaxCommandLength()
        {
            return 15;
        }
        public String ProcessCommand(String command)
        {
            return "Unknown command";
        }
    }
}
