using System;
using System.Collections.Generic;
using System.Text;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace zpg
{
    /**
     * Holds 'display' data of an object. Allows sharing same vertex buffer betveen graphic
     * objects that differ only by world matrix;
     */
    public abstract class GraphicObjectView
    {
        public GraphicObjectView(GraphicObject data)
        {
            DataSource = data;
        }

        public GraphicObject DataSource
        {
            get { return data; }
            set { data = value; }
        }

        /**
         * Perform this object 'live' actions
         */
        public void DoUpdate()
        {
            Update();
        }

        /**
         * Whatever 'life' this object has, do it here.
         */
        public abstract void Update();

        /**
         * Render this object
         */
        public void DoRendering()
        {
            data.DoRendering( this );
        }

        public Vector3 ToObjectCoordinates(Vector3 v)
        {
            Matrix iworld = world;
            iworld.Invert();
            return Vector3.TransformCoordinate(v, iworld);
        }

        public Vector3 ToWorldCoordinates(Vector3 v)
        {
            return Vector3.TransformCoordinate(v, world);
        }

        /**
         * Graphical data of the object
         */
        private GraphicObject data;

        /**
         * Convenience method
         */
        public void Translate(float x, float y, float z)
        {
            world *= Matrix.Translation(x, y, z);
        }

        /**
         * Convenience method
         */
        public void Translate(Vector3 t)
        {
            Translate(t.X, t.Y, t.Z);
        }



        /**
         * Convenience method
         */
        public void Scale(float x, float y, float z)
        {
            world *= Matrix.Scaling(x, y, z);
            
        }

        /**
         * Convenience method
         */
        public void RotateX(float angle)
        {
            world *= Matrix.RotationX(angle);
        }

        /**
         * Convenience method
         */
        public void RotateY(float angle)
        {
            world *= Matrix.RotationY(angle);
        }

        /**
         * Convenience method
         */
        public void RotateZ(float angle)
        {
            world *= Matrix.RotationZ(angle);
        }

        /**
         * Convenience method
         */
        public void RotateAxis(Vector3 axis,float angle)
        {
            world *= Matrix.RotationAxis(axis, angle);
        }


        /**
         * Fill Mode used during rendering
         */
        public FillMode FillMode
        {
            get { return fillmode; }
            set { fillmode = value; }
        }

        /**
         * World matrix for wrapped object
         */
        public Matrix Transformation
        {
            get { return world; }
            set { world = value; ; }
        }
        /**
         * Material this object is made of
         */
        public Material Material
        {
            get { return material; }
            set { material = value; }
        }
        /**
         * World matrix for wrapped object
         */
        private Matrix world = Matrix.Identity;

        /**
         * Material this object is made of
         */
        protected Material material = new Material();
        /**
         * Fill Mode used during rendering
         */
        protected FillMode fillmode = Properties.Settings.Default.globalFillMode;
    }
}
