﻿using System;
using System.IO;
using System.Data;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using D3D=Microsoft.DirectX.Direct3D;

namespace Game
{
	public class ZpgD3D : Form
	{
		#region Globalni konstanty
    	/// <summary>Pocet bodu sceny v jednom smeru.</summary>
    	private const short VELIKOST = 128;
    	
    	/// <summary>Vzdalenost bodu sceny v metrech.</summary>
    	private const short ROZTEC = 2;

        /// <summary>Pocet ruznych barev objektu.</summary>
        private const short POCETBAREV = 10;

        /// <summary>Vzdalenost bodu sceny v metrech.</summary>
        private const string MAPFILE = "mapa128x128.raw";

        //  <summary>Umisteni objektu ve scene - mapa</summary>
        private const string OBJECTFILE = "obj128x128.raw";

    	/// <summary>Rychlost pohybu v metrech za sekundu.</summary>
    	private const float RYCHLOST = 3;
    	
    	/// <summary>Vyska postavy v metrech.</summary>
    	private const float VYSKA = 1.85f;    	
    	
    	/// <summary>Delka dne (12h) v sekundach.</summary>
    	private const float DEN = 60f;

        /// <summary>Jemnost zadani vysek</summary>
        private const float HEIGHT_SCALE = 10;

        /// <summary>Dohlednost</summary>
        private const float RANGE = 128f;    
        
        /// <summary>Citlivost mysi - vyssi cislo = mensi citlivost.</summary>
        private const float CITLIVOST = 400.0f;    			
		#endregion Globalni konstanty
		
		#region Globalni promenne
//        private float debug = 0;
        
        /// <summary>azimut - rotace podle vertikalni osy - rozhlizeni.</summary>
        private float azimut = 0;

        /// <summary>elevace - rotace podle horizontalni osy - koukani nahoru a dolu.</summary>
        private float elevace = 0;
        
        /// <summary>Vektor posunu pozice stredu (moji pozice).</summary>
        private Vector3 posun = new Vector3(-(VELIKOST-1)*ROZTEC/2, -(VELIKOST-1)*ROZTEC/2, 0);

        /// <summary>Vektor smeru pohybu.</summary>
        private Vector2 smer = new Vector2(0, 0); 

        /// <summary>Vektor hodnoty pohybu.</summary>
        private Vector2 pohyb = new Vector2(0, 0); 
		
        /// <summary>Direct3D device.</summary>		
		private Device device = null;

        /// <summary> Texturka</summary>
//        private Texture texture = null;

		/// <summary>Direct3D vertx buffer.</summary>
		private VertexBuffer vertexBuffer = null;
		
		/// <summary>Direct3D index buffer.</summary>
		private IndexBuffer indexBuffer = null;		
		
		/// <summary>Vyskova mapa.</summary>
   		private float[,] heightData;

        /// <summary>Objektova mapa.</summary>
        private int[,] objectData;
        
        /// <summary>Pole vrcholu.</summary>
		private CustomVertex.PositionNormal[] verts;

        //Zmena vysky a drahy a kosinus uhlu pod kterym stoupame (klesame)
        private double zmenaVysky;
        private double zmenaDrahy;
        private double uhel;

        //Prohozeni pohledu vzhuru a dolu
        private Boolean mouseInvert = false;

        // dratovy model - zapnuto / vypnuto
        private bool dratovyModelZapnut = false;

        // objekty
        private bool objektyZapnuty = true;
        private Mesh meshObjektu;
        private int[,] poziceObjektu;
        private int pocetObjektu;
        private Texture[] texturyObjektu;
        private Texture texturaTerenu;
//        private int[,] polohaObjektu;

	    /// <summary>
        /// Pocet tiku od minuleho framu.
        /// </summary>
        private long tiku = 0;					//
        private int pocetTiku = 1;				//
        private int Frames = 0;					//
        private float snimkovaciFrekvence = 0;   // fps behem minule vteriny
        private D3D.Font text;					//
        #endregion Globalni promenne
		
		/// <summary>Hlavni konstruktor</summary>
		public ZpgD3D()
		{
            // nastaveni parametru okna
            this.Size = new Size(800, 600);
            this.Text = "ZPG Game";
            
            // pohyb mysi - event handler
            this.MouseMove += new MouseEventHandler(ZpgD3DPohybMysi);
            // stisk klavasy - event handler
            this.KeyDown += new KeyEventHandler(ZpgD3DStiskKlavesy);
            this.KeyUp += new KeyEventHandler(ZpgD3DPusteniKlavesy);
		}

		/// <summary>Nacteni vyskovych dat.</summary>       
		private bool NacistVysky() {
            heightData = new float[VELIKOST, VELIKOST];
            try
            {
                FileStream fs = new FileStream(MAPFILE, FileMode.Open, FileAccess.Read);
                BinaryReader vysky = new BinaryReader(fs);
                for (int x = 0; x < VELIKOST; x++)
                {
                    for (int y = 0; y < VELIKOST; y++)
                    {
                        heightData[VELIKOST - 1 - y, VELIKOST - 1 - x] = ((float) vysky.ReadByte() / HEIGHT_SCALE);
                    }
                }
                vysky.Close();
            }
            catch (Exception e)
            {
                MessageBox.Show("Map file not found." + e.Message);
                return false;
            }
            return true;
        }

        private bool NacistPolohuObjektu()
        {
            objectData = new int[VELIKOST-1, VELIKOST-1];
            try
            {
                FileStream fs = new FileStream(OBJECTFILE, FileMode.Open, FileAccess.Read);
                BinaryReader fsReader = new BinaryReader(fs);
                pocetObjektu = 0;
                for (int x = 0; x < VELIKOST-1; x++)
                {
                    for (int y = 0; y < VELIKOST-1; y++)
                    {
                        int objekt = (int)fsReader.ReadByte();
                        objectData[VELIKOST - 2 - y, VELIKOST - 2 - x] = objekt;
                        if (objekt > 0)
                        {
                            pocetObjektu++;
                        }
                    }
                }
                poziceObjektu = new int[pocetObjektu, 3];
                int objektCislo = 0;
                for (int x = 0; x < VELIKOST-1; x++)
                {
                    for (int y = 0; y < VELIKOST-1; y++)
                    {
                        if (objectData[y, x] > 0)
                        {
                            poziceObjektu[objektCislo, 0] = x;
                            poziceObjektu[objektCislo, 1] = y;
                            poziceObjektu[objektCislo, 2] = objectData[y, x];
                            objektCislo++;
                        }
                    }
                }

                fsReader.Close();
            }
            catch (Exception e)
            {
                MessageBox.Show("Object file not found." + e.Message);
                return false;
            }
            return true;
        }
        
        // zapise bitmapu s polohy objektu, pro ucely testovani
        private bool ZapsatPolohuObjektu()
        {
            objectData = new int[VELIKOST - 1, VELIKOST - 1];
            objectData[68, 63] = 32;
            objectData[20, 30] = 24;
            objectData[40, 100] = 43;
            objectData[44, 100] = 41;
            objectData[46, 100] = 48;
            objectData[40, 104] = 33;
            objectData[44, 104] = 51;
            objectData[100, 80] = 50;
            objectData[96, 82] = 21;
            objectData[110, 86] = 23;
            objectData[100, 90] = 35;
            objectData[89, 104] = 21;
            try
            {
                FileStream fs = new FileStream(OBJECTFILE, FileMode.Create, FileAccess.Write);
                BinaryWriter fsReader = new BinaryWriter(fs);
                for (int x = 0; x < VELIKOST -1 ; x++)
                {
                    for (int y = 0; y < VELIKOST -1; y++)
                    {
                        fsReader.Write((byte)objectData[VELIKOST - 2 - y, VELIKOST - 2 - x]);
                    }
                }
                fsReader.Close();
            }
            catch (Exception e)
            {
                MessageBox.Show("Can't write object file: " + e.Message);
                return false;
            }
            return true;
        }

        private void NacistObjekty()
        {
            meshObjektu = Mesh.FromFile("cube.x", MeshFlags.Managed, device);
            meshObjektu = meshObjektu.Clone(meshObjektu.Options.Value, CustomVertex.PositionNormal.Format, device);
            meshObjektu.ComputeNormals();

            texturyObjektu = new Texture[POCETBAREV];
            for (int i = 0; i < POCETBAREV; i++)
            {
                texturyObjektu[i] = TextureLoader.FromFile(device, Application.StartupPath + @"\barva" + i + ".bmp");
            }
            texturaTerenu = TextureLoader.FromFile(device, Application.StartupPath + @"\teren.bmp");
        }

		
		/// <summary>Nastaveni parametru grafiky.</summary>
		public bool InicializaceGrafiky()
		{
			try
			{
				#region nastaveni parametru
				PresentParameters presentParams = new PresentParameters();
				presentParams.Windowed=true; // We don't want to run fullscreen
				presentParams.SwapEffect = SwapEffect.Discard; // Discard the frames 
				presentParams.EnableAutoDepthStencil = true; // Turn on a Depth stencil
				presentParams.AutoDepthStencilFormat = DepthFormat.D16; // And the stencil format			
				// zjisteni a nastaveni schopnosti graficke karty
                Caps caps = Manager.GetDeviceCaps(0, Microsoft.DirectX.Direct3D.DeviceType.Hardware);
                CreateFlags flags;
                if (caps.DeviceCaps.SupportsHardwareTransformAndLight)
                    flags = CreateFlags.HardwareVertexProcessing;
                else
                    flags = CreateFlags.SoftwareVertexProcessing;
                // vytvorit D3D device
                device = new Microsoft.DirectX.Direct3D.Device(0,
                	Microsoft.DirectX.Direct3D.DeviceType.Hardware, this, flags, presentParams);
				#endregion nastaveni parametru
				
				#region nastaveni device
                // vypnout cull - uvidime predni i zadni cast trojuhelniku
				device.RenderState.CullMode = Cull.None;
				// zapnout ZBuffer
				device.RenderState.ZBufferEnable = true;
				// zkontrolovat zapnita svetla
				device.RenderState.Lighting = true;
				// naplnit VB
				vertexBuffer = new VertexBuffer(typeof(CustomVertex.PositionNormal), VELIKOST*VELIKOST,
					device, Usage.WriteOnly, CustomVertex.PositionNormal.Format, Pool.Default);
				this.VytvoritVertexBuffer(vertexBuffer);
				// naplnit IB
				indexBuffer = new IndexBuffer(typeof(short), (VELIKOST-1)*(VELIKOST-1)*6,
					device, Usage.WriteOnly, Pool.Default);
				this.VytvoritIndexBuffer(indexBuffer);				
                #endregion nastaveni device

                try
                {
                    NacistObjekty();
                }
                catch (Exception e)
                {
                    MessageBox.Show("Error loading mesh files." + e.Message);
                    return false;
                }

				return true;
			}
            catch (DirectXException) {
                MessageBox.Show("Selhala inicializace direct3D.", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }
		}
		
        
		/// <summary>Vraci norm vektorovy soucin vektoru tvorenych z vrcholu a jeho dvou sousedu.</summary>
		private Vector3 ProVypocetNormalVB(int i, int j, int k)
		{
			Vector3 smer1 = new Vector3(verts[i].X - verts[k].X, verts[i].Y - verts[k].Y, verts[i].Z - verts[k].Z);
            Vector3 smer2 = new Vector3(verts[j].X - verts[k].X, verts[j].Y - verts[k].Y, verts[j].Z - verts[k].Z);
            Vector3 vektorSoucin = new Vector3();
            smer1.Normalize();
            smer2.Normalize();
            vektorSoucin = Vector3.Cross(smer1,smer2);
            vektorSoucin.Normalize();
            return vektorSoucin;
		}
		
		/// <summary>Naplneni vertex bufferu daty.</summary>
		public void VytvoritVertexBuffer(object sender)
		{
            VertexBuffer vb = (VertexBuffer)sender;
			// zamknuti
			verts = (CustomVertex.PositionNormal[])vb.Lock(0,0);
			// naplneni
            for (int x=0;x< VELIKOST;x++)
            {
            	for (int y=0; y< VELIKOST;y++)
            	{
            		verts[x+y*VELIKOST].X=ROZTEC*x;
            		verts[x+y*VELIKOST].Y=ROZTEC*y;
            		verts[x+y*VELIKOST].Z=heightData[x,y];
                }
            }
            Vector3 normala = new Vector3();
            for (int x=1;x< VELIKOST-1;x++)
            {
            	for (int y=1; y< VELIKOST-1;y++)
            	{
            		normala.X = 0;
            		normala.Y = 0;
            		normala.Z = 0;
            		int i = x+y*VELIKOST;
            		normala.Add(ProVypocetNormalVB(i-1-128, i-1	 , i));
            		normala.Add(ProVypocetNormalVB(i-1	, i-1+128, i));
            		normala.Add(ProVypocetNormalVB(i-1+128, i+128	 , i));
            		normala.Add(ProVypocetNormalVB(i+128	, i+1+128, i));
            		normala.Add(ProVypocetNormalVB(i+1+128, i+1	 , i));
            		normala.Add(ProVypocetNormalVB(i+1	, i+1-128, i));
            		normala.Add(ProVypocetNormalVB(i+1-128, i-128	 , i));
            		normala.Add(ProVypocetNormalVB(i-128	, i-1-128, i));
            		normala.Normalize();
            		verts[x+y*VELIKOST].Nx=normala.X;
            		verts[x+y*VELIKOST].Ny=normala.Y;
            		verts[x+y*VELIKOST].Nz=normala.Z;
                }
            }
            
			// odemknuti
			vb.Unlock();
		}
		
		/// <summary>Naplneni index bufferu daty.</summary>
		public void VytvoritIndexBuffer(object sender)
		{
            IndexBuffer indexb = (IndexBuffer)sender;
			// zamknuti
			short[] indices = (short[])indexb.Lock(0,0);
			// naplneni
            for (int x=0;x< VELIKOST-1;x++)
            {
            	for (int y=0; y< VELIKOST-1;y++)
            	{
                	indices[(x+y*(VELIKOST-1))*6] = (short) ((x+1)+(y+1)*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+1] = (short) ((x+1)+y*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+2] = (short) (x+y*VELIKOST);
                		
                	indices[(x+y*(VELIKOST-1))*6+3] = (short) ((x+1)+(y+1)*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+4] = (short) (x+y*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+5] = (short) (x+(y+1)*VELIKOST);
                }
            }
			// odemknuti
			indexb.Unlock();
		}		
		
		/// <summary>Nastaveni matic projekce, sveta a kamery.</summary>
		private void NastaveniMatic()
		{
			// world posun sveta dle nasi pozice
		    device.Transform.World = Matrix.Translation(posun);  
			
			// nastaveni kamery a natoceni pohledu
			device.Transform.View = Matrix.LookAtLH(new Vector3(0,0,0), new Vector3(0,1,0), new Vector3(0,0,1));
            device.Transform.View *= Matrix.RotationY(azimut);
            device.Transform.View *= Matrix.RotationX(elevace);    
            
			// perspectivni transformace
			// + nastaveni hlobky viditelnosti
			float aspect = (float)this.ClientSize.Width / (float)this.ClientSize.Height;
            device.Transform.Projection = Matrix.PerspectiveFovLH((float)Math.PI / 4, aspect, 0.1f, RANGE);	         
		}
		
		/// <summary>Nastaveni vlastnosti svetel.</summary>
		private void NastaveniSvetel()
		{
			System.Drawing.Color col = System.Drawing.Color.White;
			// nastavit material.
			// v jednom case muze byt pouzit pouze jeden material
            D3D.Material mtrl = new D3D.Material();
            mtrl.Diffuse = col;
			mtrl.Ambient = col;
			device.Material = mtrl;
			
			// nastavit slunce.
			// v nastaveni grafiky je treba enablovat D3DRS_LIGHTING
			device.Lights[0].Type = LightType.Directional;
			device.Lights[0].Diffuse = System.Drawing.Color.White;
            device.Lights[0].Direction =
				new Vector3((float)-Math.Cos((Math.PI*Environment.TickCount) / (1000*DEN)),
				            0.0f, (float)-Math.Sin((Math.PI*Environment.TickCount) / (1000*DEN)));
			// zaponout svetlo
			device.Lights[0].Enabled = true;
			
			// nastavit mesic.			
			device.Lights[1].Type = LightType.Directional;
			device.Lights[1].Diffuse = System.Drawing.Color.DarkGray;
			device.Lights[1].Direction =
				new Vector3((float)Math.Cos((Math.PI*Environment.TickCount) / (1000*DEN)),
				            0.2f, (float)Math.Sin((Math.PI*Environment.TickCount) / (1000*DEN)));
			device.Lights[1].Enabled = true;

            // vypnovaci mod
            device.RenderState.FillMode = FillMode.Solid;
            // textura
            device.SetTexture(0, texturaTerenu);
        }
		
		/// <summary>Vypocet posunu po terenu.</summary>
		private void Pohyb()
		{
		 Vector3 V1 = new Vector3(0, 0, 0);
         Vector3 V2 = new Vector3(0, 0, 0);
         Vector3 V3 = new Vector3(0, 0, 0);
         Vector3 oldPozice = posun;
         Vector2 pohybRovinny;
         float vzorkovani = 5f;
			
            // vypocet uplynuleho casu
            long noveTiku = Environment.TickCount;
            float uplynulyCas = noveTiku - tiku;
            tiku = noveTiku;
          
            // vypocet posunu ve vodorovnem smeru

            pohyb.X = (smer.X * (float)Math.Sin(azimut) + smer.Y * (float)Math.Cos(azimut));
            pohyb.Y = (smer.X * (float)Math.Cos(azimut) - smer.Y * (float)Math.Sin(azimut));
            
            pohyb.Normalize();

            pohybRovinny.X = pohyb.X * RYCHLOST * uplynulyCas / 1000 *vzorkovani;
            pohybRovinny.Y = pohyb.Y * RYCHLOST * uplynulyCas / 1000 *vzorkovani;
            posun.X -= pohybRovinny.X;
            posun.Y -= pohybRovinny.Y;

            // kontrola ze nevypadneme z mapy
            if (posun.X > -1 ||
                posun.X < (1 - (VELIKOST - 1) * ROZTEC) ||
                posun.Y > -1 ||
                posun.Y < (1 - (VELIKOST - 1) * ROZTEC))
            {
                    posun.X = oldPozice.X;
                    posun.Y = oldPozice.Y;
                    return;
            }
            // vrcholy trojuhelnika nad kterym se nachazime
            int pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+1+(Math.Truncate(-posun.Y/ROZTEC)+1)*VELIKOST);
            V1.X = verts[pomocna].X;
            V1.Y = verts[pomocna].Y;
            V1.Z = verts[pomocna].Z;
            if ((-posun.X/ROZTEC - Math.Truncate(-posun.X/ROZTEC)) < (-posun.Y/ROZTEC - Math.Truncate(-posun.Y/ROZTEC))) {
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST);
            	V2.X = verts[pomocna].X;
            	V2.Y = verts[pomocna].Y;
            	V2.Z = verts[pomocna].Z;
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+(Math.Truncate(-posun.Y/ROZTEC)+1)*VELIKOST); 
            	V3.X = verts[pomocna].X;
            	V3.Y = verts[pomocna].Y;
            	V3.Z = verts[pomocna].Z;
            }
            else {
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+1+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST); 
            	V2.X = verts[pomocna].X;
            	V2.Y = verts[pomocna].Y;
            	V2.Z = verts[pomocna].Z;
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST); 
            	V3.X = verts[pomocna].X;
            	V3.Y = verts[pomocna].Y;
            	V3.Z = verts[pomocna].Z;
            }
            
            // vypocet z - souradnice pruseciku plochy trojuhelnika
            // s rovnobezkou s osou z prochazejici nasi pozici
            posun.Z = - VYSKA - V1.Z
            	- (((V2.Y-V1.Y)*(V3.Z-V1.Z)-(V2.Z-V1.Z)*(V3.Y-V1.Y))*(V1.X+posun.X)
            	+ ((V3.X-V1.X)*(V2.Z-V1.Z)-(V3.Z-V1.Z)*(V2.X-V1.X))*(V1.Y+posun.Y))
            	/((V2.X-V1.X)*(V3.Y-V1.Y)-(V2.Y-V1.Y)*(V3.X-V1.X));

//            posun.Z = -VYSKA - V1.Z - (((V2.Z - V1.Z) * ((V1.X + posun.X) / ROZTEC) + (V3.Z - V2.Z) * ((V1.Y + posun.Y) / ROZTEC)) / 2);

            //Vypocet sklonu 
            zmenaVysky = Math.Abs(posun.Z - oldPozice.Z);
            zmenaDrahy = (float)(Math.Sqrt(Math.Pow(pohybRovinny.X,2) + Math.Pow(pohybRovinny.Y,2)));
          
            uhel = (Math.Cos(Math.Asin(zmenaVysky / zmenaDrahy)));
            if (uhel.Equals(double.NaN)) uhel=1;

            // vraceni pozorovatele o kousek zpet
            posun.X += pohybRovinny.X - (pohybRovinny.X/vzorkovani * (float)uhel);    
            posun.Y += pohybRovinny.Y - (pohybRovinny.Y/vzorkovani * (float)uhel);

            // kontrola ze nevypadneme z mapy
            if (posun.X > -1) posun.X = -1;
            if (posun.X < (1 - (VELIKOST - 1) * ROZTEC)) posun.X = 1 - (VELIKOST - 1) * ROZTEC;
            if (posun.Y > -1) posun.Y = -1;
            if (posun.Y < (1 - (VELIKOST - 1) * ROZTEC)) posun.Y = 1 - (VELIKOST - 1) * ROZTEC;

            // kontrola kolize s objekty
            for (int i = 0; i < pocetObjektu; i++)
            {
                float presah = 0.1f;
                if (-posun.X > (poziceObjektu[i, 0] - presah) * ROZTEC && -posun.X < (poziceObjektu[i, 0] + 1 + presah) * ROZTEC &&
                    -posun.Y > (poziceObjektu[i, 1] - presah) * ROZTEC && -posun.Y < (poziceObjektu[i, 1] + 1 + presah) * ROZTEC)
                {
                    posun.X = oldPozice.X;
                    posun.Y = oldPozice.Y;
                }
            }

            // vrcholy trojuhelnika nad kterym se nachazime po uprave vzdalenosti
            pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + 1 + (Math.Truncate(-posun.Y / ROZTEC) + 1) * VELIKOST);
            V1.X = verts[pomocna].X;
            V1.Y = verts[pomocna].Y;
            V1.Z = verts[pomocna].Z;
            if ((-posun.X / ROZTEC - Math.Truncate(-posun.X / ROZTEC)) < (-posun.Y / ROZTEC - Math.Truncate(-posun.Y / ROZTEC)))
            {
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + Math.Truncate(-posun.Y / ROZTEC) * VELIKOST);
                V2.X = verts[pomocna].X;
                V2.Y = verts[pomocna].Y;
                V2.Z = verts[pomocna].Z;
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + (Math.Truncate(-posun.Y / ROZTEC) + 1) * VELIKOST);
                V3.X = verts[pomocna].X;
                V3.Y = verts[pomocna].Y;
                V3.Z = verts[pomocna].Z;
            }
            else
            {
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + 1 + Math.Truncate(-posun.Y / ROZTEC) * VELIKOST);
                V2.X = verts[pomocna].X;
                V2.Y = verts[pomocna].Y;
                V2.Z = verts[pomocna].Z;
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + Math.Truncate(-posun.Y / ROZTEC) * VELIKOST);
                V3.X = verts[pomocna].X;
                V3.Y = verts[pomocna].Y;
                V3.Z = verts[pomocna].Z;
            }

            // novy prepocet vysky v novem X a Y
            posun.Z = -VYSKA - V1.Z
                - (((V2.Y - V1.Y) * (V3.Z - V1.Z) - (V2.Z - V1.Z) * (V3.Y - V1.Y)) * (V1.X + posun.X)
                + ((V3.X - V1.X) * (V2.Z - V1.Z) - (V3.Z - V1.Z) * (V2.X - V1.X)) * (V1.Y + posun.Y))
                / ((V2.X - V1.X) * (V3.Y - V1.Y) - (V2.Y - V1.Y) * (V3.X - V1.X));            
        }

        // vykresli dratovy model
        private void NakresliDratovyModel()
        {
            // teren se vykresli s dratovou siti            
            device.RenderState.FillMode = FillMode.WireFrame;
            device.SetStreamSource(0, vertexBuffer, 0);
            device.VertexFormat = CustomVertex.PositionOnly.Format;
            device.Indices = indexBuffer;
            device.DrawIndexedPrimitives(PrimitiveType.TriangleList, 0, 0, VELIKOST * VELIKOST, 0, (VELIKOST - 1) * (VELIKOST - 1) * 2);

            // a zbyle objekty zase vylnovane
            device.RenderState.FillMode = FillMode.Solid;
        }

        // nakresli objekty
        private void NakresliObjekty()
        {
            for (int i = 0; i < pocetObjektu; i++)
            {
                // nastaveni umisteni a rotace objektu
                device.Transform.World = Matrix.Identity;
                // posun na stred ctverce
                device.Transform.World *= Matrix.Translation(new Vector3(1f, 1f, 1f));
                // natazeni do vysky
                device.Transform.World *= Matrix.Scaling(new Vector3(1f, 1f, (poziceObjektu[i, 2] / 10)));
 
                // vypocet nejnizzi vysky pro umisteni objektu
                device.Transform.World *= Matrix.Translation(posun);
                float h1 = heightData[poziceObjektu[i, 0], poziceObjektu[i, 1]];
                float h2 = heightData[poziceObjektu[i, 0] + 1 , poziceObjektu[i, 1]];
                float h3 = heightData[poziceObjektu[i, 0], poziceObjektu[i, 1] + 1];
                float h4 = heightData[poziceObjektu[i, 0] + 1, poziceObjektu[i, 1] + 1];
                float h = Math.Min(Math.Min(h1, h2), Math.Min(h3, h4));
                device.Transform.World *= Matrix.Translation(poziceObjektu[i, 0] * ROZTEC,
                                                             poziceObjektu[i, 1] * ROZTEC,                                                           
                                                             h);
                // vykresleni objektu
                device.SetTexture(0, texturyObjektu[(poziceObjektu[i, 2] % 10)]);
                for (int j = 0; j < meshObjektu.GetAttributeTable().Length; ++j)
                {
                    meshObjektu.DrawSubset(j);
                }
            }

        }

		/// <summary>Vykresleni sceny.</summary>
		private void Render()
		{
			if (device == null) return;
			// vycistit backbuffer na modro a nastavit depth buffer
			device.Clear(ClearFlags.Target | ClearFlags.ZBuffer, Color.Black, 1f, 0);

            // zacatek sestaveni sceny
			device.BeginScene();

            
            // nastavit svetla a materialy
			NastaveniSvetel();

            // posun po terenu
            Pohyb();

            // nastavit matice sveta, pohledu a projekce
			NastaveniMatic();
			// trojuhelnikova mapa
			device.SetStreamSource(0, vertexBuffer, 0);
			device.VertexFormat = CustomVertex.PositionNormal.Format;
			device.Indices = indexBuffer;
            device.DrawIndexedPrimitives(PrimitiveType.TriangleList, 0, 0, VELIKOST*VELIKOST, 0, (VELIKOST-1)*(VELIKOST-1)*2);
            // dratovy model
            if (dratovyModelZapnut)
            {
                NakresliDratovyModel();
            }
		    // zobrazit framerate - nakonec, aby byl navrch
		    FrameRate();
            text.DrawText(null,string.Format("Framerate : {0:0.00} fps", snimkovaciFrekvence), 
            	new Rectangle(10, 10,760,200),Microsoft.DirectX.Direct3D.DrawTextFormat.Right, Color.White);
            text.DrawText(null, string.Format("Pozice : X= {0:0.00}, Y= {1:0.00}", posun.X, posun.Y),
               new Rectangle(10, 30, 760, 200), Microsoft.DirectX.Direct3D.DrawTextFormat.Right, Color.White);

            // kresleni objektu
            if (objektyZapnuty)
            {
                NakresliObjekty();
            }
            
            // konec sestaveni
			device.EndScene();
			// zobrazeni vysledku
			device.Present();
		}
		
		/// <summary>Vypocet framerate - ulozeno do promenne snimkovaci frekvence.</summary>
		private void FrameRate()
		{
		    Frames++;
            if (Math.Abs(Environment.TickCount - pocetTiku) > 1000) {
            	snimkovaciFrekvence = (float)Frames * 1000 / Math.Abs(Environment.TickCount - pocetTiku);
                pocetTiku = Environment.TickCount;
                Frames = 0;
            }
		}
		
        /// <summary>Stisk klavasy - event handler.</summary>
        private void ZpgD3DStiskKlavesy(object sender, KeyEventArgs e) {            
            switch (e.KeyCode) {
                case Keys.Escape:
                    // ukonci aplikaci
                    this.Close();
                    break;
                case (Keys.M):
                    dratovyModelZapnut = !dratovyModelZapnut;
                    break;
                case (Keys.W):
                case (Keys.Up):
                    // pohyb vpred
                    smer.X = 1;
                    break;
                case Keys.S:
                case Keys.Down:
                    // pohyb vzad
                    smer.X = -1;                    
                    break;
                case Keys.D:
                case Keys.Right:
                    // pohyb vpravo
                    smer.Y = -1;
                    break; 
                case Keys.A:
                case Keys.Left:
                    // pohyb vlevo
                    smer.Y = 1;
                    break;
                case Keys.U:
                    //prohozeni divani se nahoru a dolu
                    mouseInvert = !mouseInvert;
                    break;
                case Keys.Space:
                    // nastavit vychozi hodnoty
                    posun.X = -VELIKOST/ROZTEC*ROZTEC;
                    posun.Y = -VELIKOST/ROZTEC*ROZTEC;
                    azimut = 0;
                    elevace = 0;
                    break;
                default:
                    break;
            }
        }

        private void ZpgD3DPusteniKlavesy(object sender, KeyEventArgs e) 
        {
            switch (e.KeyCode) {
                case (Keys.W):
                case (Keys.Up):
                    // pohyb vpred
                    smer.X = 0;
                    break;
                case Keys.S:
                case Keys.Down:
                    // pohyb vzad
                    smer.X = 0;
                    break;
                case Keys.D:
                case Keys.Right:
                    // pohyb vpravo
                    smer.Y = 0;
                    break; 
                case Keys.A:
                case Keys.Left:
                    // pohyb vlevo
                    smer.Y = 0;
                    break;
                default:
                    break;
            }  
        }	
        
        /// <summary>Pohyb mysi - event handler.</summary>
        private void ZpgD3DPohybMysi(object sender, MouseEventArgs e) {
            // vypocet stredu okna
            Size s = this.ClientSize;
            int cx = s.Width / 2;
            int cy = s.Height / 2;

            // kontrola zda se hyblo s mysi
            if (e.X == cx && e.Y == cy) return;

            // aktualizace uhlu
            azimut += (float)(cx - e.X) / CITLIVOST;

            if (mouseInvert)
            {
                elevace -= (float)(cy - e.Y) / CITLIVOST;
            }
            else
            {
                elevace += (float)(cy - e.Y) / CITLIVOST;
            }
            // kontrola rozsahu uhlu
            if (azimut > 2 * Math.PI) azimut -= 2 * (float)Math.PI;
            if (elevace > Math.PI / 2) elevace = (float)Math.PI / 2;
            else if (elevace < -Math.PI / 2) elevace = -(float)Math.PI / 2;

            // vypocitat pozici stredu rel. k desktopu z pozice stredu rel. k soucasnemu oknu
            Point center = PointToScreen(new Point(cx, cy));
            // vycentrovat kurzor
            Cursor.Position = center;
        }              
     
        /// <summary>Vlastni spusteni.</summary>
        private void Run() {
            // schovat kurzor mysi
            Cursor.Hide();
            // zjistit velikost okna a spocitat stred
            Size s = this.ClientSize;
            int cx = s.Width / 2;
            int cy = s.Height / 2;
            // vypocitat pozici stredu rel. k desktopu z pozice stredu rel. k soucasnemu oknu
            Point center = PointToScreen(new Point(cx, cy));
            // vycentrovat kurzor
            Cursor.Position = center;
            // nastavit cas
            tiku = Environment.TickCount;
            // priprava zobrazovaneho textu - musi byt po inicializaci device
            System.Drawing.Font systemfont = new System.Drawing.Font("Arial", 10f, FontStyle.Regular);
            text = new D3D.Font(device,systemfont);            

            // behova smycka
            while (this.Created) {
                Render();
                Application.DoEvents(); // melo by byt posledni ve smycce
            }

            // zobrazit kurzor mysi
            Cursor.Show();
            // zrusit zinicializovane
            if (vertexBuffer != null) vertexBuffer.Dispose();
            if (indexBuffer != null) indexBuffer.Dispose();
            if (device != null) device.Dispose();
        }		
        
        /// <summary>Start aplikace.</summary>
		static void Main() 
		{
			// vytvoreni okna aplikace
			ZpgD3D form = new ZpgD3D();
            // zapis bitmapy s objekty, pro testovaci ucely
            form.ZapsatPolohuObjektu();
            // nacteni vyskove mapy a polohy objektu
            if (form.NacistVysky() && form.NacistPolohuObjektu())
            {
                // zobrazeni okna
                form.Show();
                // inicializace Direct 3D
                form.Text += " - Načítáni dat.";
                if (form.InicializaceGrafiky())
                {
                    // vlastni spusteni
                    form.Text = "ZPG Game";
                    form.Run();
                }
            }
            form.Dispose();
		}
	}
}
