﻿using System;
using System.IO;
using System.Data;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using D3D=Microsoft.DirectX.Direct3D;

namespace Game
{
	public class ZpgD3D : Form
	{
		#region Globalni konstanty
    	/// <summary>Pocet bodu sceny v jednom smeru.</summary>
    	private const short VELIKOST = 128;
    	
    	/// <summary>Vzdalenost bodu sceny v metrech.</summary>
    	private const short ROZTEC = 2;
    	
    	/// <summary>Rychlost pohybu v metrech za sekundu.</summary>
    	private const float RYCHLOST = 3;
    	
    	/// <summary>Vyska postavy v metrech.</summary>
    	private const float VYSKA = 1.85f;    	
    	
    	/// <summary>Delka dne (12h) v sekundach.</summary>
    	private const float DEN = 60f;

        /// <summary>Jemnost zadani vysek</summary>
        private const short HEIGHT_SCALE = 10;
    
        
        /// <summary>Citlivost mysi - vyssi cislo = mensi citlivost.</summary>
        private const float CITLIVOST = 400.0f;    			
		#endregion Globalni konstanty
		
		#region Globalni promenne
        private float debug = 0;
        
        /// <summary>azimut - rotace podle vertikalni osy - rozhlizeni.</summary>
        private float azimut = 0;

        /// <summary>elevace - rotace podle horizontalni osy - koukani nahoru a dolu.</summary>
        private float elevace = 0;
        
        /// <summary>Vektor posunu pozice stredu (moji pozice).</summary>
        private Vector3 posun = new Vector3(-(VELIKOST-1)*ROZTEC/2, -(VELIKOST-1)*ROZTEC/2, 0);
        
        /// <summary>Vektor smeru pohybu.</summary>
        private Vector2 pohyb = new Vector2(0, 0); 
		
        /// <summary>Direct3D device.</summary>		
		private Device device = null;

        /// <summary> Texturka</summary>
//        private Texture texture = null;

		/// <summary>Direct3D vertx buffer.</summary>
		private VertexBuffer vertexBuffer = null;
		
		/// <summary>Direct3D index buffer.</summary>
		private IndexBuffer indexBuffer = null;		
		
		/// <summary>Vyskova mapa.</summary>
		private float[,] heightData;
		
		/// <summary>Pole vrcholu.</summary>
		private CustomVertex.PositionNormal[] verts;

        //Zmena vysky a drahy a kosinus uhlu pod kterym stoupame (klesame)
        private double zmenaVysky;
        private double zmenaDrahy;
        private double uhel;

        //Prohozeni pohledu vzhuru a dolu
        private Boolean mouseInvert = false;

	    /// <summary>
        /// Pocet tiku od minuleho framu.
        /// </summary>
        private long tiku = 0;					//
        private int pocetTiku = 1;				//
        private int Frames = 0;					//
        private float snimkovaciFrekvence = 0;   // fps behem minule vteriny
        private D3D.Font text;					//
        #endregion Globalni promenne
		
		/// <summary>Hlavni konstruktor</summary>
		public ZpgD3D()
		{
            // nastaveni parametru okna
            this.Size = new Size(800, 600);
            this.Text = "ZPG Game";
            
            // pohyb mysi - event handler
            this.MouseMove += new MouseEventHandler(ZpgD3DPohybMysi);
            // stisk klavasy - event handler
            this.KeyDown += new KeyEventHandler(ZpgD3DStiskKlavesy);			
		}
		
		/// <summary>Nacteni vyskovych dat.</summary>       
		private void NacistVysky() {
             heightData = new float[VELIKOST,VELIKOST];
             FileStream fs = new FileStream("mapa128x128.raw",FileMode.Open, FileAccess.Read);
             BinaryReader r = new BinaryReader(fs);
             
             for (int i = 0;i< VELIKOST;i++) {
             	for (int y=0; y< VELIKOST; y++) {
                    heightData[VELIKOST-1-y,VELIKOST-1-i] = (int)(r.ReadByte()/HEIGHT_SCALE);;
                }
             }
             r.Close();
        }   

		
		/// <summary>Nastaveni parametru grafiky.</summary>
		public bool InicializaceGrafiky()
		{
			try
			{
				#region nastaveni parametru
				PresentParameters presentParams = new PresentParameters();
				presentParams.Windowed=true; // We don't want to run fullscreen
				presentParams.SwapEffect = SwapEffect.Discard; // Discard the frames 
				presentParams.EnableAutoDepthStencil = true; // Turn on a Depth stencil
				presentParams.AutoDepthStencilFormat = DepthFormat.D16; // And the stencil format			
				// zjisteni a nastaveni schopnosti graficke karty
                Caps caps = Manager.GetDeviceCaps(0, Microsoft.DirectX.Direct3D.DeviceType.Hardware);
                CreateFlags flags;
                if (caps.DeviceCaps.SupportsHardwareTransformAndLight)
                    flags = CreateFlags.HardwareVertexProcessing;
                else
                    flags = CreateFlags.SoftwareVertexProcessing;
                // vytvorit D3D device
                device = new Microsoft.DirectX.Direct3D.Device(0,
                	Microsoft.DirectX.Direct3D.DeviceType.Hardware, this, flags, presentParams);
				#endregion nastaveni parametru
				
				#region nastaveni device
                // vypnout cull - uvidime predni i zadni cast trojuhelniku
				device.RenderState.CullMode = Cull.None;
				// zapnout ZBuffer
				device.RenderState.ZBufferEnable = true;
				// zkontrolovat zapnita svetla
				device.RenderState.Lighting = true;
				// naplnit VB
				vertexBuffer = new VertexBuffer(typeof(CustomVertex.PositionNormal), VELIKOST*VELIKOST,
					device, Usage.WriteOnly, CustomVertex.PositionNormal.Format, Pool.Default);
				this.VytvoritVertexBuffer(vertexBuffer);
				// naplnit IB
				indexBuffer = new IndexBuffer(typeof(short), (VELIKOST-1)*(VELIKOST-1)*6,
					device, Usage.WriteOnly, Pool.Default);
				this.VytvoritIndexBuffer(indexBuffer);				
                #endregion nastaveni device

				return true;
			}
            catch (DirectXException) {
                MessageBox.Show("Selhala inicializace direct3D.", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }
		}
		
        
		/// <summary>Vraci norm vektorovy soucin vektoru tvorenych z vrcholu a jeho dvou sousedu.</summary>
		private Vector3 ProVypocetNormalVB(int i, int j, int k)
		{
			Vector3 smer1 = new Vector3(verts[i].X - verts[k].X, verts[i].Y - verts[k].Y, verts[i].Z - verts[k].Z);
            Vector3 smer2 = new Vector3(verts[j].X - verts[k].X, verts[j].Y - verts[k].Y, verts[j].Z - verts[k].Z);
            Vector3 vektorSoucin = new Vector3();
            smer1.Normalize();
            smer2.Normalize();
            vektorSoucin = Vector3.Cross(smer1,smer2);
            vektorSoucin.Normalize();
            return vektorSoucin;
		}
		
		/// <summary>Naplneni vertex bufferu daty.</summary>
		public void VytvoritVertexBuffer(object sender)
		{
            VertexBuffer vb = (VertexBuffer)sender;
			// zamknuti
			verts = (CustomVertex.PositionNormal[])vb.Lock(0,0);
			// naplneni
            for (int x=0;x< VELIKOST;x++)
            {
            	for (int y=0; y< VELIKOST;y++)
            	{
            		verts[x+y*VELIKOST].X=ROZTEC*x;
            		verts[x+y*VELIKOST].Y=ROZTEC*y;
            		verts[x+y*VELIKOST].Z=heightData[x,y];
                }
            }
            Vector3 normala = new Vector3();
            for (int x=1;x< VELIKOST-1;x++)
            {
            	for (int y=1; y< VELIKOST-1;y++)
            	{
            		normala.X = 0;
            		normala.Y = 0;
            		normala.Z = 0;
            		int i = x+y*VELIKOST;
            		normala.Add(ProVypocetNormalVB(i-1-128, i-1	 , i));
            		normala.Add(ProVypocetNormalVB(i-1	, i-1+128, i));
            		normala.Add(ProVypocetNormalVB(i-1+128, i+128	 , i));
            		normala.Add(ProVypocetNormalVB(i+128	, i+1+128, i));
            		normala.Add(ProVypocetNormalVB(i+1+128, i+1	 , i));
            		normala.Add(ProVypocetNormalVB(i+1	, i+1-128, i));
            		normala.Add(ProVypocetNormalVB(i+1-128, i-128	 , i));
            		normala.Add(ProVypocetNormalVB(i-128	, i-1-128, i));
            		normala.Normalize();
            		verts[x+y*VELIKOST].Nx=normala.X;
            		verts[x+y*VELIKOST].Ny=normala.Y;
            		verts[x+y*VELIKOST].Nz=normala.Z;
                }
            }
            
			// odemknuti
			vb.Unlock();
		}
		
		/// <summary>Naplneni index bufferu daty.</summary>
		public void VytvoritIndexBuffer(object sender)
		{
            IndexBuffer indexb = (IndexBuffer)sender;
			// zamknuti
			short[] indices = (short[])indexb.Lock(0,0);
			// naplneni
            for (int x=0;x< VELIKOST-1;x++)
            {
            	for (int y=0; y< VELIKOST-1;y++)
            	{
                	indices[(x+y*(VELIKOST-1))*6] = (short) ((x+1)+(y+1)*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+1] = (short) ((x+1)+y*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+2] = (short) (x+y*VELIKOST);
                		
                	indices[(x+y*(VELIKOST-1))*6+3] = (short) ((x+1)+(y+1)*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+4] = (short) (x+y*VELIKOST);
                	indices[(x+y*(VELIKOST-1))*6+5] = (short) (x+(y+1)*VELIKOST);
                }
            }
			// odemknuti
			indexb.Unlock();
		}		
		
		/// <summary>Nastaveni matic projekce, sveta a kamery.</summary>
		private void NastaveniMatic()
		{
			// world posun sveta dle nasi pozice
		    device.Transform.World = Matrix.Translation(posun);  
			
			// nastaveni kamery a natoceni pohledu
			device.Transform.View = Matrix.LookAtLH(new Vector3(0,0,0), new Vector3(0,1,0), new Vector3(0,0,1));
            device.Transform.View *= Matrix.RotationY(azimut);
            device.Transform.View *= Matrix.RotationX(elevace);    
            
			// perspectivni transformace
			// + nastaveni hlobky viditelnosti
			float aspect = (float)this.ClientSize.Width / (float)this.ClientSize.Height;
            device.Transform.Projection = Matrix.PerspectiveFovLH((float)Math.PI / 4, aspect, 1f, 128.0f);	         
		}
		
		/// <summary>Nastaveni vlastnosti svetel.</summary>
		private void NastaveniSvetel()
		{
			System.Drawing.Color col = System.Drawing.Color.White;
			// nastavit material.
			// v jednom case muze byt pouzit pouze jeden material
            D3D.Material mtrl = new D3D.Material();
            mtrl.Diffuse = col;
			mtrl.Ambient = col;
			device.Material = mtrl;
			
			// nastavit slunce.
			// v nastaveni grafiky je treba enablovat D3DRS_LIGHTING
			device.Lights[0].Type = LightType.Directional;
			device.Lights[0].Diffuse = System.Drawing.Color.LightGreen;
            device.Lights[0].Direction =
				new Vector3((float)-Math.Cos((Math.PI*Environment.TickCount) / (1000*DEN)),
				            0.0f, (float)-Math.Sin((Math.PI*Environment.TickCount) / (1000*DEN)));
			// zaponout svetlo
			device.Lights[0].Enabled = true;
			
			// nastavit mesic.			
			device.Lights[1].Type = LightType.Directional;
			device.Lights[1].Diffuse = System.Drawing.Color.DarkOliveGreen;
			device.Lights[1].Direction =
				new Vector3((float)Math.Cos((Math.PI*Environment.TickCount) / (1000*DEN)),
				            0.2f, (float)Math.Sin((Math.PI*Environment.TickCount) / (1000*DEN)));
			device.Lights[1].Enabled = true;

            // vypnovaci mod
            device.RenderState.FillMode = FillMode.Solid ;
		}
		
		/// <summary>Vypocet posunu po terenu.</summary>
		private void Pohyb()
		{
		 Vector3 V1 = new Vector3(0, 0, 0);
         Vector3 V2 = new Vector3(0, 0, 0);
         Vector3 V3 = new Vector3(0, 0, 0);
         Vector3 oldPozice = posun;
         Vector2 pohybRovinny;
         float vzorkovani = 5f;
			
            // vypocet uplynuleho casu
            long noveTiku = Environment.TickCount;
            float uplynulyCas = noveTiku - tiku;
            tiku = noveTiku;
          
            // vypocet posunu ve vodorovnem smeru
            pohyb.Normalize();

            pohybRovinny.X = pohyb.X * RYCHLOST * uplynulyCas / 1000 *vzorkovani;
            pohybRovinny.Y = pohyb.Y * RYCHLOST * uplynulyCas / 1000 *vzorkovani;
            posun.X -= pohybRovinny.X;
            posun.Y -= pohybRovinny.Y;
            
            // kontrola ze nevypadneme z mapy
            if (posun.X > -1) posun.X = -1;
            if (posun.X < (1-(VELIKOST-1)*ROZTEC)) posun.X = 1-(VELIKOST-1)*ROZTEC;
            if (posun.Y > -1) posun.Y = -1;
            if (posun.Y < (1-(VELIKOST-1)*ROZTEC)) posun.Y = 1-(VELIKOST-1)*ROZTEC;

            // vrcholy trojuhelnika nad kterym se nachazime
            int pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+1+(Math.Truncate(-posun.Y/ROZTEC)+1)*VELIKOST);
            V1.X = verts[pomocna].X;
            V1.Y = verts[pomocna].Y;
            V1.Z = verts[pomocna].Z;
            if ((-posun.X/ROZTEC - Math.Truncate(-posun.X/ROZTEC)) < (-posun.Y/ROZTEC - Math.Truncate(-posun.Y/ROZTEC))) {
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST);
            	V2.X = verts[pomocna].X;
            	V2.Y = verts[pomocna].Y;
            	V2.Z = verts[pomocna].Z;
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+(Math.Truncate(-posun.Y/ROZTEC)+1)*VELIKOST); 
            	V3.X = verts[pomocna].X;
            	V3.Y = verts[pomocna].Y;
            	V3.Z = verts[pomocna].Z;
            }
            else {
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+1+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST); 
            	V2.X = verts[pomocna].X;
            	V2.Y = verts[pomocna].Y;
            	V2.Z = verts[pomocna].Z;
            	pomocna = (int) (Math.Truncate(-posun.X/ROZTEC)+Math.Truncate(-posun.Y/ROZTEC)*VELIKOST); 
            	V3.X = verts[pomocna].X;
            	V3.Y = verts[pomocna].Y;
            	V3.Z = verts[pomocna].Z;
            }
            
            // vypocet z - souradnice pruseciku plochy trojuhelnika
            // s rovnobezkou s osou z prochazejici nasi pozici
            posun.Z = - VYSKA - V1.Z
            	- (((V2.Y-V1.Y)*(V3.Z-V1.Z)-(V2.Z-V1.Z)*(V3.Y-V1.Y))*(V1.X+posun.X)
            	+ ((V3.X-V1.X)*(V2.Z-V1.Z)-(V3.Z-V1.Z)*(V2.X-V1.X))*(V1.Y+posun.Y))
            	/((V2.X-V1.X)*(V3.Y-V1.Y)-(V2.Y-V1.Y)*(V3.X-V1.X));

//            posun.Z = -VYSKA - V1.Z - (((V2.Z - V1.Z) * ((V1.X + posun.X) / ROZTEC) + (V3.Z - V2.Z) * ((V1.Y + posun.Y) / ROZTEC)) / 2);

            //Vypocet sklonu 
            zmenaVysky = Math.Abs(posun.Z - oldPozice.Z);
            zmenaDrahy = (float)(Math.Sqrt(Math.Pow(pohybRovinny.X,2) + Math.Pow(pohybRovinny.Y,2)));
          
            uhel = (Math.Cos(Math.Asin(zmenaVysky / zmenaDrahy)));
            if (uhel.Equals(double.NaN)) uhel=1;

            // vraceni pozorovatele o kousek zpet
            posun.X += pohybRovinny.X - (pohybRovinny.X/vzorkovani * (float)uhel);    
            posun.Y += pohybRovinny.Y - (pohybRovinny.Y/vzorkovani * (float)uhel);

            // vrcholy trojuhelnika nad kterym se nachazime po uprave vzdalenosti
            pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + 1 + (Math.Truncate(-posun.Y / ROZTEC) + 1) * VELIKOST);
            V1.X = verts[pomocna].X;
            V1.Y = verts[pomocna].Y;
            V1.Z = verts[pomocna].Z;
            if ((-posun.X / ROZTEC - Math.Truncate(-posun.X / ROZTEC)) < (-posun.Y / ROZTEC - Math.Truncate(-posun.Y / ROZTEC)))
            {
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + Math.Truncate(-posun.Y / ROZTEC) * VELIKOST);
                V2.X = verts[pomocna].X;
                V2.Y = verts[pomocna].Y;
                V2.Z = verts[pomocna].Z;
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + (Math.Truncate(-posun.Y / ROZTEC) + 1) * VELIKOST);
                V3.X = verts[pomocna].X;
                V3.Y = verts[pomocna].Y;
                V3.Z = verts[pomocna].Z;
            }
            else
            {
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + 1 + Math.Truncate(-posun.Y / ROZTEC) * VELIKOST);
                V2.X = verts[pomocna].X;
                V2.Y = verts[pomocna].Y;
                V2.Z = verts[pomocna].Z;
                pomocna = (int)(Math.Truncate(-posun.X / ROZTEC) + Math.Truncate(-posun.Y / ROZTEC) * VELIKOST);
                V3.X = verts[pomocna].X;
                V3.Y = verts[pomocna].Y;
                V3.Z = verts[pomocna].Z;
            }

            // novy prepocet vysky v novem X a Y
            posun.Z = -VYSKA - V1.Z
                - (((V2.Y - V1.Y) * (V3.Z - V1.Z) - (V2.Z - V1.Z) * (V3.Y - V1.Y)) * (V1.X + posun.X)
                + ((V3.X - V1.X) * (V2.Z - V1.Z) - (V3.Z - V1.Z) * (V2.X - V1.X)) * (V1.Y + posun.Y))
                / ((V2.X - V1.X) * (V3.Y - V1.Y) - (V2.Y - V1.Y) * (V3.X - V1.X));            
        }
		
		/// <summary>Vykresleni sceny.</summary>
		private void Render()
		{
			if (device == null) return;
			// vycistit backbuffer na modro a nastavit depth buffer
			device.Clear(ClearFlags.Target | ClearFlags.ZBuffer, Color.Black, 1f, 0);

            // zacatek sestaveni sceny
			device.BeginScene();
			// nastavit svetla a materialy
			NastaveniSvetel();
			// posun po terenu
			Pohyb();
			// nastavit matice sveta, pohledu a projekce
			NastaveniMatic();
			// trojuhelnikova mapa
			device.SetStreamSource(0, vertexBuffer, 0);
			device.VertexFormat = CustomVertex.PositionNormal.Format;
			device.Indices = indexBuffer;
            device.DrawIndexedPrimitives(PrimitiveType.TriangleList, 0, 0, VELIKOST*VELIKOST, 0, (VELIKOST-1)*(VELIKOST-1)*2);
			
		    // zobrazit framerate - nakonec, aby byl navrch
		    FrameRate();
            text.DrawText(null,string.Format("Framerate : {0:0.00} fps", snimkovaciFrekvence), 
            	new Rectangle(10, 10,760,200),Microsoft.DirectX.Direct3D.DrawTextFormat.Right, Color.White);
            text.DrawText(null, string.Format("Nova : {0:0.00} fps", debug),
                new Rectangle(10, 30, 760, 200), Microsoft.DirectX.Direct3D.DrawTextFormat.Right, Color.White);
            text.DrawText(null, string.Format("Vyska : {0:0.00} fps", posun.Z),
                new Rectangle(10, 50, 760, 200), Microsoft.DirectX.Direct3D.DrawTextFormat.Right, Color.White);
            // konec sestaveni
			device.EndScene();
			// zobrazeni vysledku
			device.Present();
            // vynulovat pohyb
            pohyb.X = 0;
            pohyb.Y = 0;			
		}
		
		/// <summary>Vypocet framerate - ulozeno do promenne snimkovaci frekvence.</summary>
		private void FrameRate()
		{
		    Frames++;
            if (Math.Abs(Environment.TickCount - pocetTiku) > 1000) {
            	snimkovaciFrekvence = (float)Frames * 1000 / Math.Abs(Environment.TickCount - pocetTiku);
                pocetTiku = Environment.TickCount;
                Frames = 0;
            }
		}
		
        /// <summary>Stisk klavasy - event handler.</summary>
        private void ZpgD3DStiskKlavesy(object sender, KeyEventArgs e) {
            switch (e.KeyCode) {
                case Keys.Escape:
                    // ukonci aplikaci
                    this.Close();
                    break;
                case (Keys.W):
                    // pohyb vpred
                    pohyb.X = (float) Math.Sin(azimut);
                    pohyb.Y = (float) Math.Cos(azimut);
                    break;
                case Keys.S:
                    // pohyb vzad
                    pohyb.X = -(float) Math.Sin(azimut);
                    pohyb.Y = -(float) Math.Cos(azimut);                    
                    break;
                case Keys.D:
                    // pohyb vpravo
                    pohyb.X = -(float) Math.Cos(azimut);
                    pohyb.Y = (float) Math.Sin(azimut);
                    break; 
                case Keys.A:
                    // pohyb vlevo
                    pohyb.X = (float) Math.Cos(azimut);
                    pohyb.Y = -(float) Math.Sin(azimut);
                    break;
                case Keys.U:
                    //prohozeni divani se nahoru a dolu
                    if (mouseInvert)
                    {
                        mouseInvert = false;
                    }
                    else
                    {
                        mouseInvert = true;
                    }
                    break;
                case Keys.Space:
                    // nastavit vychozi hodnoty
                    posun.X = -VELIKOST/ROZTEC*ROZTEC;
                    posun.Y = -VELIKOST/ROZTEC*ROZTEC;
                    azimut = 0;
                    elevace = 0;
                    break;
                default:
                    break;
            }
        }   
        
        /// <summary>Pohyb mysi - event handler.</summary>
        private void ZpgD3DPohybMysi(object sender, MouseEventArgs e) {
            // vypocet stredu okna
            Size s = this.ClientSize;
            int cx = s.Width / 2;
            int cy = s.Height / 2;

            // kontrola zda se hyblo s mysi
            if (e.X == cx && e.Y == cy) return;

            // aktualizace uhlu
            azimut += (float)(cx - e.X) / CITLIVOST;

            if (mouseInvert)
            {
                elevace -= (float)(cy - e.Y) / CITLIVOST;
            }
            else
            {
                elevace += (float)(cy - e.Y) / CITLIVOST;
            }
            // kontrola rozsahu uhlu
            if (azimut > 2 * Math.PI) azimut -= 2 * (float)Math.PI;
            if (elevace > Math.PI / 2) elevace = (float)Math.PI / 2;
            else if (elevace < -Math.PI / 2) elevace = -(float)Math.PI / 2;

            // vypocitat pozici stredu rel. k desktopu z pozice stredu rel. k soucasnemu oknu
            Point center = PointToScreen(new Point(cx, cy));
            // vycentrovat kurzor
            Cursor.Position = center;
        }              
     
        /// <summary>Vlastni spusteni.</summary>
        private void Run() {
            // schovat kurzor mysi
            Cursor.Hide();
            // zjistit velikost okna a spocitat stred
            Size s = this.ClientSize;
            int cx = s.Width / 2;
            int cy = s.Height / 2;
            // vypocitat pozici stredu rel. k desktopu z pozice stredu rel. k soucasnemu oknu
            Point center = PointToScreen(new Point(cx, cy));
            // vycentrovat kurzor
            Cursor.Position = center;
            // nastavit cas
            tiku = Environment.TickCount;
            // priprava zobrazovaneho textu - musi byt po inicializaci device
            System.Drawing.Font systemfont = new System.Drawing.Font("Arial", 10f, FontStyle.Regular);
            text = new D3D.Font(device,systemfont);            

            // behova smycka
            while (this.Created) {
                Render();
                Application.DoEvents(); // melo by byt posledni ve smycce
            }

            // zobrazit kurzor mysi
            Cursor.Show();
            // zrusit zinicializovane
            if (vertexBuffer != null) vertexBuffer.Dispose();
            if (indexBuffer != null) indexBuffer.Dispose();
            if (device != null) device.Dispose();
        }		
        
        /// <summary>Start aplikace.</summary>
		static void Main() 
		{
			// vytvoreni okna aplikace
			ZpgD3D form = new ZpgD3D();
            // zobrazeni okna
            form.Show();
            // nacteni vyskove mapy
            form.NacistVysky();
            // inicializace Direct 3D
            form.Text += " - Načítáni dat.";
            if (form.InicializaceGrafiky()){
            	// vlastni spusteni
            	form.Text = "ZPG Game";
                form.Run();
            }   
		}
	}
}
